<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Subjects;
use App\Models\Subtopics;
use App\Models\Questions;
use App\Models\QuizResult;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Auth;
use DB;
use Illuminate\Support\Str;
use App\Models\QuizSession;
use App\Models\QuestionMetadata;
use Illuminate\Support\Facades\Log;
class DashboardController extends Controller
{
    //
    public function index()
    {
        $subjects = Subjects::withCount(['subtopics', 'questions'])
    ->orderBy('created_at', 'desc')
    ->get();

        $subtopic = Subtopics::get();
        return response()->json([
            'success' => true,
            'subjects' => $subjects,
            'data' => $subjects, // For backward compatibility
            'summary' => [
                'total_subjects' => $subjects->count(),
                'total_subtopics' => count($subtopic),
                'total_questions' => $subjects->sum('questions_count'),
                'auto_generate_active' => $subjects->where('auto', 1)->count(),
            ]
        ]);
    }
    
    public function subtopic(Request $request){
        $subtopics = Subtopics::where('subject_id', $request->subjectId)->get();
        return response()->json([
            'success' => true,
            'data' => $subtopics
        ]);
    }
    public function schools(){
        $schools = Institutions::all();
        return response()->json([
            'success' => true,
            'data' => $schools
        ]);
    }
    public function question_store(Request $request)
    {
        
        
        // Define validation rules
        $rules = [
            'question_type' => 'required|in:mcq,viva',
            'question' => 'required|string',
            'subject_id' => 'required|exists:subjects,id',
            'difficulty' => 'required|in:Easy,Medium,Hard',
            'subtopic_id' => 'required'
        ];

        // Add conditional validation rules based on question type
        if ($request->question_type === 'mcq') {
            $rules['options'] = 'required|array|min:2';
            $rules['options.*'] = 'required|string';
            $rules['correct_answer'] = 'required|string';
        } else {
            $rules['viva_type'] = 'required|string';
            $rules['answer'] = 'required|string';
        }

        // Validate request
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'error' => $validator->errors(),
            ], 422);
        }
        if($request->userData){
            $userData = json_decode($request->userData, true);
        }
        
       
        // Prepare question data
        $questionData = [
            'question' => $request->question,
            'subject_id' => $request->subject_id,
            'difficulty' => $request->difficulty,
            'school_id' => $userData['institution'],
            'uploaded_by' => $userData['id'],
            'subtopic_id' => $request->subtopic_id,
            'question_type' => $request->question_type === 'mcq' ? 'multiple_choice' : 'open_ended',
        ];

        // Handle answer based on question type
        if ($request->question_type === 'mcq') {
            
            $questionData['answer'] = json_encode([
                'options' => $request->options,
                'correct_answer' => $request->correct_answer
            ]);
        } else {
            $questionData['answer'] = $request->answer;
        }

        // Create question
        try {
            $question = Questions::create($questionData);
            
            // If viva question, store viva_type in a separate table or as metadata
            if ($request->question_type === 'viva') {
                $question->metadata()->create([
                    'viva_type' => $request->viva_type
                ]);
            }

            return response()->json([
                'message' => 'Question created successfully',
                'data' => $question
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to create question',
                'message' => $e->getMessage()
            ], 500);
        }
    }
    public function getQuestions(Request $request)
    {
        try {
            // Validate query parameters
            $validator = Validator::make($request->all(), [
                'subject' => 'nullable|integer|exists:subjects,id',
                'subtopic' => 'nullable|integer|exists:sub_topics,id',
                'difficulty' => 'nullable|string|in:Easy,Medium,Hard',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid query parameters',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Build the query for multiple-choice questions
            $query = Questions::query()
                ->where('question_type', 'multiple_choice');

            // Apply filters if provided
            if ($request->has('subject') && $request->subject !== 'all') {
                $query->where('subject_id', $request->subject);
            }

            if ($request->has('subtopic') && $request->subtopic !== 'all') {
                $query->where('subtopic_id', $request->subtopic);
            }

            if ($request->has('difficulty') && $request->difficulty !== 'all') {
                $query->where('difficulty', $request->difficulty);
            }

            // Fetch questions with related subject and subtopic
            $questions = $query->with(['subject', 'subtopic'])
                ->get([
                    'id',
                    'question',
                    'answer',
                    'subject_id',
                    'subtopic_id',
                    'difficulty'
                ]);

            // Check if no questions were found
            if ($questions->isEmpty()) {
                return response()->json([
                    'status' => 'success',
                    'message' => 'No questions found',
                    'data' => []
                ], 200);
            }

            // Transform questions to match front-end format
            $transformedQuestions = $questions->map(function ($question) {
                // Decode JSON answer or provide a default structure
                $answer = is_array($question->answer)
                    ? $question->answer
                    : (json_decode($question->answer, true) ?? ['options' => [], 'correct_answer' => '']);
            
                // Ensure options and correct_answer exist
                $options = $answer['options'] ?? [];
                $correctAnswer = $answer['correct_answer'] ?? '';
            
                return [
                    'id' => $question->id,
                    'question' => $question->question,
                    'options' => is_array($options) ? $options : [], // Ensure options is an array
                    'correct_answer' => is_string($correctAnswer) ? $correctAnswer : '', // Ensure correct_answer is a string
                    'explanation' => '', // No explanation column in DB
                    'subject' => $question->subject ? $question->subject->name : null,
                    'subtopic' => $question->subtopic ? $question->subtopic->name : null,
                    'difficulty' => $question->difficulty
                ];
            });

            return response()->json([
                'status' => 'success',
                'message' => 'Questions retrieved successfully',
                'data' => $transformedQuestions
            ], 200);

        } catch (\Exception $e) {
            \Log::error('Error fetching questions: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch questions',
                'error' => $e->getMessage() // Remove in production
            ], 500);
        }
    }

  /**
     * Start a new quiz session
     */
    public function startQuiz(Request $request)
    {
        try {
            // Validate request parameters
            $validator = Validator::make($request->all(), [
                'subject' => 'nullable|integer|exists:subjects,id',
                'subtopic' => 'nullable|integer|exists:sub_topics,id',
                'difficulty' => 'nullable|string|in:Easy,Medium,Hard',
                'question_count' => 'required|integer|min:1|max:50',
                'time_limit' => 'nullable|integer|min:1|max:180'
            ]);
    
            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid parameters',
                    'errors' => $validator->errors()
                ], 422);
            }
    
            // Build the query for multiple-choice questions
            $query = Questions::query()
                ->where('question_type', 'multiple_choice');
    
            $subjectId = null;
            $subtopicId = null;
    
            // Apply filters
            if ($request->filled('subject') && $request->subject !== 'all') {
                $subjectId = $request->subject;
                $query->where('subject_id', $subjectId);
            }
    
            if ($request->filled('subtopic') && $request->subtopic !== 'all') {
                $subtopicId = $request->subtopic;
                $query->where('subtopic_id', $subtopicId);
            }
    
            if ($request->filled('difficulty') && $request->difficulty !== 'all') {
                $query->where('difficulty', $request->difficulty);
            }
    
            // First: Count how many questions are available
            $availableCount = $query->count();
    
            if ($availableCount === 0) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No questions found for the selected criteria'
                ], 404);
            }
    
            // Determine actual number of questions to use
            $requestedCount = $request->question_count;
            $actualCount = min($requestedCount, $availableCount); // Use all if fewer than requested
    
            // Get random questions (up to available count)
            $questions = $query->with(['subject', 'subtopic'])
                ->inRandomOrder()
                ->limit($actualCount)
                ->get([
                    'id',
                    'question',
                    'answer',
                    'subject_id',
                    'subtopic_id',
                    'difficulty',
                    
                ]);
    
            // Optional: Inform user if we adjusted the count
            $adjustedMessage = $actualCount < $requestedCount
                ? " (Only {$availableCount} questions available, using all)"
                : "";
    
            // Create quiz session
            $sessionId = Str::uuid()->toString();
    
            $quizData = $questions->map(function ($question) {
                $answer = is_array($question->answer)
                    ? $question->answer
                    : (json_decode($question->answer, true) ?? ['options' => [], 'correct_answer' => '']);
    
                return [
                    'question_id' => $question->id,
                    'correct_answer' => $answer['correct_answer'] ?? '',
                    'explanation' => $answer['explanation'] ?? ''
                ];
            })->toArray();
    
            $timeLimit = $request->time_limit ?? 60;
            $expiresAt = now()->addMinutes($timeLimit + 10);
    
            QuizSession::create([
                'session_id' => $sessionId,
                'user_id' => $request->user()->id,
                'questions_data' => json_encode($quizData),
                'total_questions' => $questions->count(),
                'time_limit' => $request->time_limit,
                'status' => 'active',
                'subject_id' => $subjectId,
                'difficulty' => $request->difficulty !== 'all' ? $request->difficulty : null,
                'started_at' => now(),
                'expires_at' => $expiresAt
            ]);
            $user = auth()->user();
            $today = Carbon::today();

            if ($user->last_study_date) {
                $lastStudy = Carbon::parse($user->last_study_date);
        
                if ($lastStudy->isToday()) {
        
                    $newStreak = $user->streak;
                } elseif ($lastStudy->isYesterday()) {
                   
                    $newStreak = $user->streak + 1;
                } else {
                    
                    $newStreak = 1;
                }
            } else {
              
                $newStreak = 1;
            }
        
           
            $user->streak = $newStreak;
            $user->last_study_date = $today;
            $user->save();
    
            $questionsForFrontend = $questions->map(function ($question) {
                $answer = is_array($question->answer)
                    ? $question->answer
                    : (json_decode($question->answer, true) ?? ['options' => []]);
    
                return [
                    'id' => $question->id,
                    'question' => $question->question,
                    'options' => is_array($answer['options']) ? $answer['options'] : [],
                    'subject' => $question->subject ? $question->subject->name : null,
                    'subtopic' => $question->subtopic ? $question->subtopic->name : null,
                    'difficulty' => $question->difficulty,
                    'explanation' => $question->explanation
                ];
            });
    
            return response()->json([
                'status' => 'success',
                'message' => "Quiz started with {$actualCount} question(s){$adjustedMessage}",
                'session_id' => $sessionId,
                'questions' => $questionsForFrontend,
                'total_questions' => $actualCount,
                'time_limit' => $request->time_limit,
                
            ], 200);
    
        } catch (\Exception $e) {
            Log::error('Error starting quiz: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
    
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to start quiz session'
            ], 500);
        }
    }

    /**
     * Check a single answer in Learning Mode
     */
    public function checkAnswer(Request $request)
    {
        try {
            // Validate request parameters
            $validator = Validator::make($request->all(), [
                'session_id' => 'required|string',
                'question_id' => 'required|integer|exists:questions,id',
                'selected_answer' => 'required|string'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid parameters',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Find quiz session
            $session = QuizSession::where('session_id', $request->session_id)
                ->where('user_id', auth()->id())
                ->where('status', 'active')
                ->first();

            if (!$session) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid or expired quiz session'
                ], 404);
            }

            // Check if session has expired
            if ($session->expires_at && $session->expires_at < now()) {
                $session->update(['status' => 'expired']);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Quiz session has expired'
                ], 410);
            }

            // Get question data
            $question = Questions::find($request->question_id);
            if (!$question) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Question not found'
                ], 404);
            }

            // Decode answer data
            $answerData = is_array($question->answer)
                ? $question->answer
                : (json_decode($question->answer, true) ?? ['correct_answer' => '', 'explanation' => '']);

            $correctAnswer = $answerData['correct_answer'] ?? '';
            $explanation = $question->explanation ?? '';

            // Check if answer is correct
            $isCorrect = $request->selected_answer === $correctAnswer;

            // Update session with user answer (for analytics)
            $questionsData = json_decode($session->questions_data, true);
            $userAnswers = json_decode($session->user_answers, true) ?? [];
            $userAnswers[] = [
                'question_id' => $request->question_id,
                'selected_answer' => $request->selected_answer,
                'is_correct' => $isCorrect
            ];

            $session->update([
                'user_answers' => json_encode($userAnswers)
            ]);

            return response()->json([
                'status' => 'success',
                'is_correct' => $isCorrect,
                'correct_answer' => $correctAnswer,
                'explanation' => $explanation
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error checking answer: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to check answer'
            ], 500);
        }
    }

    /**
     * Submit quiz answers and get results
     */
    public function submitQuiz(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'session_id' => 'required',
                'answers' => 'required|array',
                'time_spent' => 'required|numeric'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid submission data',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Find quiz session
            $session = QuizSession::where('session_id', $request->session_id)
                ->where('user_id', auth()->id())
                ->where('status', 'active')
                ->first();

            if (!$session) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid or expired quiz session'
                ], 404);
            }

            // Check if session has expired
            if ($session->expires_at < now()) {
                $session->update(['status' => 'expired']);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Quiz session has expired'
                ], 410);
            }

            // Properly decode the questions data
            $rawData = $session->questions_data;
            $questionsData = json_decode($rawData, true);

            // Handle double encoding
            while (is_string($questionsData)) {
                $questionsData = json_decode($questionsData, true);
            }

            // Ensure valid data
            if (!is_array($questionsData)) {
                Log::error('Failed to decode questions data', ['raw' => $rawData]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid questions data format'
                ], 500);
            }

            Log::info('Questions data after decoding:', ['data' => $questionsData]);

            $userAnswers = $request->answers;

            // Get full question details from database
            $questionIds = collect($questionsData)->pluck('question_id')->toArray();

            Log::info('Question IDs extracted:', ['ids' => $questionIds]);

            $questions = DB::table('questions')
                ->whereIn('id', $questionIds)
                ->get()
                ->keyBy('id');

            Log::info('Questions retrieved from DB:', ['count' => $questions->count(), 'questions' => $questions->toArray()]);

            // Calculate results
            $totalQuestions = count($questionsData);
            $correctAnswers = 0;
            $detailedResults = [];

            foreach ($questionsData as $index => $questionData) {
                $questionId = $questionData['question_id'];
                $question = $questions->get($questionId);

                if (!$question) {
                    continue;
                }

                // Decode the answer JSON from the database
                $answerData = json_decode($question->answer, true);

                // Find user answer for this question
                $userAnswerData = collect($userAnswers)->firstWhere('question_id', $questionId);
                $userAnswer = $userAnswerData['selected_answer'] ?? null;

                $correctAnswer = $answerData['correct_answer'];
                $isCorrect = $userAnswer === $correctAnswer;

                if ($isCorrect) {
                    $correctAnswers++;
                }

                $detailedResults[] = [
                    'question_id' => $questionId,
                    'question_text' => $question->question,
                    'options' => $answerData['options'] ?? [],
                    'user_answer' => $userAnswer,
                    'correct_answer' => $correctAnswer,
                    'is_correct' => $isCorrect,
                    'explanation' => $question->explanation ?? $question->explanation ?? '',
                    'difficulty' => $question->difficulty,
                    'question_type' => $question->question_type
                ];

                Log::info('Processed question:', [
                    'question_id' => $questionId,
                    'question_found' => !is_null($question),
                    'question_text' => $question ? $question->question : 'NOT FOUND',
                    'answer_data' => $answerData
                ]);
            }

            $score = $totalQuestions > 0 ? round(($correctAnswers / $totalQuestions) * 100, 2) : 0;

            // Update session as completed
            $session->update([
                'status' => 'completed',
                'user_answers' => json_encode($userAnswers),
                'score' => $score,
                'correct_answers' => $correctAnswers,
                'completed_at' => now(),
                'time_spent' => $request->time_spent
            ]);

            return response()->json([
                'status' => 'success',
                'message' => 'Quiz submitted successfully',
                'results' => [
                    'session_id' => $request->session_id,
                    'total_questions' => $totalQuestions,
                    'correct_answers' => $correctAnswers,
                    'score' => $score,
                    'detailed_results' => $detailedResults,
                    'completed_at' => now()->toDateTimeString(),
                    'time_spent' => $request->time_spent
                ]
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error submitting quiz: ' . $e->getMessage(), [
                'session_id' => $request->session_id ?? 'unknown',
                'user_id' => auth()->id(),
                'error' => $e->getTraceAsString()
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to submit quiz'
            ], 500);
        }
    }
    /**
     * Get quiz session status (for resuming or checking)
     */
    public function getQuizSession(Request $request, $sessionId)
    {
        try {
            $session = QuizSession::where('session_id', $sessionId)
                ->where('user_id', auth()->id())
                ->first();

            if (!$session) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Quiz session not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'session' => [
                    'session_id' => $session->session_id,
                    'status' => $session->status,
                    'total_questions' => $session->total_questions,
                    'time_limit' => $session->time_limit,
                    'started_at' => $session->started_at,
                    'expires_at' => $session->expires_at,
                    'score' => $session->score,
                    'completed_at' => $session->completed_at
                ]
            ], 200);

        } catch (\Exception $e) {
            \Log::error('Error fetching quiz session: ' . $e->getMessage());

            return response()->json([
                'status' => 'error',
                'message' => 'Failed to fetch quiz session'
            ], 500);
        }
    }





    public function quiz_result(Request $request){
       $quiz = DB::table('quiz_result')->insert([
            'user_id' => $request->userData['id'],
            'total_questions' => $request->totalQuestions,
            'correct_answers' => $request->correctAnswers,
            'quiz_type' => 'mcq',
            'score' => $request->score,
            'institution_id' => $request->userData['institution'],
            'level' => $request->userData['year_of_study'],
            'time_spent' => $request->timeSpent,
            'created_at' => now()
        ]);
        if($quiz){
            return response()->json([
                'status' => 'success',
                'message' => 'Quiz submitted successfully',
                
            ], 200);

        }

    }
       /**
     * Fetch distinct viva types
     */
    public function getVivaTypes()
    {
        try {
            $vivaTypes = QuestionMetadata::distinct()->pluck('viva_type')->filter()->values();

            return response()->json([
                'success' => true,
                'message' => 'Viva types retrieved successfully',
                'data' => $vivaTypes
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error fetching viva types: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch viva types'
            ], 500);
        }
    }
    /**
     * Fetch open-ended (viva) questions with filters
     */
    public function essay(Request $request)
    {
        try {
            // Validate request parameters
            $validated = $request->validate([
                'subject' => 'nullable|string|max:255',
                'subtopic' => 'nullable|string|max:255',
                'institution' => 'nullable|string|max:255',
                'difficulty' => 'nullable|in:Easy,Medium,Hard',
                'viva_type' => 'nullable|string|max:255',
                'search' => 'nullable|string|max:255',
                'per_page' => 'nullable|integer|min:1|max:100',
                'page' => 'nullable|integer|min:1'
            ]);

            // Start building the query
            $query = Questions::query()
                ->with(['subject:id,name', 'subtopic:id,name', 'institution:id,name', 'user:id,name'])
                ->leftJoin('question_metadata', 'questions.id', '=', 'question_metadata.question_id')
                ->where('question_type', 'open_ended');

            // Apply filters if provided
            if ($request->filled('subject')) {
                $query->where('subject_id', $validated['subject']);
            }

            if ($request->filled('subtopic')) {
                $query->where('subtopic_id', $validated['subtopic']);
            }

            if ($request->filled('institution')) {
                $query->where('school_id', $validated['institution']);
            }

            if ($request->filled('difficulty')) {
                $query->where('difficulty', $validated['difficulty']);
            }

            if ($request->filled('viva_type')) {
                $query->where('question_metadata.viva_type', $validated['viva_type']);
            }

            // Apply search filter
            if ($request->filled('search')) {
                $searchTerm = '%' . $validated['search'] . '%';
                $query->where(function ($q) use ($searchTerm) {
                    $q->where('questions.question', 'LIKE', $searchTerm)
                      ->orWhere('questions.answer', 'LIKE', $searchTerm)
                      ->orWhereHas('subject', function ($subQuery) use ($searchTerm) {
                          $subQuery->where('name', 'LIKE', $searchTerm);
                      })
                      ->orWhereHas('subtopic', function ($subQuery) use ($searchTerm) {
                          $subQuery->where('name', 'LIKE', $searchTerm);
                      })
                      ->orWhereHas('institution', function ($subQuery) use ($searchTerm) {
                          $subQuery->where('name', 'LIKE', $searchTerm);
                      });
                });
            }

            // Order by creation date (newest first)
            $query->orderBy('questions.created_at', 'desc');

            // Pagination
            $perPage = $validated['per_page'] ?? 10;
            $questions = $query->select([
                'questions.*',
                'question_metadata.viva_type'
            ])->paginate($perPage);

            // Transform the data for frontend consumption
            $transformedQuestions = $questions->map(function ($question) {
                return [
                    'id' => $question->id,
                    'question' => $question->question,
                    'answer' => json_decode($question->answer, true)['correct_answer'] ?? $question->answer,
                    'difficulty' => $question->difficulty,
                    'subject' => $question->subject?->name ?? 'N/A',
                    'subject_id' => $question->subject_id,
                    'subtopic' => $question->subtopic?->name ?? null,
                    'subtopic_id' => $question->subtopic_id,
                    'school' => $question->institution?->name ?? 'N/A',
                    'institution' => $question->institution?->name ?? 'N/A',
                    'institution_id' => $question->institution_id,
                    'viva_type' => $question->viva_type ?? null,
                    'uploadedBy' => $question->user?->name ?? 'Unknown',
                    'uploaded_by_id' => $question->user_id,
                    'uploadedAt' => $question->created_at?->format('M d, Y') ?? 'N/A',
                    'created_at' => $question->created_at,
                    'updated_at' => $question->updated_at,
                ];
            });

            // Prepare response data
            $responseData = [
                'success' => true,
                'message' => 'Questions fetched successfully',
                'questions' => $transformedQuestions,
                'pagination' => [
                    'current_page' => $questions->currentPage(),
                    'per_page' => $questions->perPage(),
                    'total' => $questions->total(),
                    'last_page' => $questions->lastPage(),
                    'from' => $questions->firstItem(),
                    'to' => $questions->lastItem(),
                ],
                'filters_applied' => [
                    'subject' => $request->get('subject'),
                    'subtopic' => $request->get('subtopic'),
                    'institution' => $request->get('institution'),
                    'difficulty' => $request->get('difficulty'),
                    'viva_type' => $request->get('viva_type'),
                    'search' => $request->get('search'),
                ],
                'total_count' => $questions->total()
            ];

            return response()->json($responseData, 200);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            Log::error('Error fetching questions: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred while fetching questions',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
public function profile(Request $request)
{
    $user = $request->user();
    $user->load('school');
    
    $profile = [
        'name' => $user->name,
        'email' => $user->email,
        'year' => $user->year ?? '',
        'institution' => $user->institution_id,
        'university' => $user->school->name ?? '',
        'role' => $user->role,
        'premium_status' => $user->is_premium,
        'bio' => $user->bio ?? 'Medical student passionate about learning and sharing knowledge...',
        'location' => $user->location ?? 'Jos, Plateau State, Nigeria',
        'is_in_trial' => $user->is_in_trial && $user->trial_start && $user->trial_start->diffInDays(now()) < 4,
        'trial_days_left' => $user->trial_start ? max(0, 4 - $user->trial_start->diffInDays(now())) : 0,
        'trial_end_date' => $user->trial_start ? $user->trial_start->addDays(4)->format('M j, Y') : null
    ];

    return response()->json([
        'success' => true,
        'data' => $profile
    ]);
}
// public function profile(Request $request)
//     {
//         $user = $request->user();
//         $user->load('school');
        
//         $profile = [
//             'name' => $user->name,
//             'email' => $user->email,
//             'year' => $user->year ?? '',
//             'university' => $user->school->name ?? '',
//             'bio' => $user->bio ?? 'Medical student passionate about learning and sharing knowledge...',
//             'location' => $user->location ?? 'Jos, Plateau State, Nigeria',
//         ];

//         return response()->json([
//             'success' => true,
//             'data' => $profile
//         ]);
//     }

    public function userStats(Request $request)
    {
        
        $user = $request->user();
        $userId = $user->id;

        try {
            // Get basic stats
            $stats = [
                'questionsUploaded' => Questions::where('uploaded_by', $userId)->count(),
                'questionsAnswered' => QuizSession::where('user_id', $userId)->sum('total_questions'),
                'studyStreak' =>  $user->streak,
                'totalPoints' => QuizSession::where('user_id', $userId)->sum('correct_answers')
            ];

            // Get recent activity (last 10 quiz attempts)
            $recentActivity = QuizSession::where('user_id', $userId)
    ->whereNotNull('time_limit')
    ->with([
        'user:id,name',
        'institution:id,name',
        'subject:id,name',
    ])
    ->orderBy('created_at', 'desc')
    ->limit(10)
    ->get()
    ->map(function ($result) {
        return [
            'id' => $result->id,
            'type' => 'quiz_completed',
            'title' => ucfirst($result->quiz_type) . ' Quiz Completed',
            'description' => "Scored {$result->correct_answers}/{$result->total_questions} points",
            'score' => intval($result->score),
            'total_questions' => $result->total_questions,
            'accuracy' => $result->total_questions > 0 
                ? round(($result->correct_answers / $result->total_questions) * 100, 1) 
                : 0,
            'time_spent' => $result->time_spent,
            'level' => $result->level,
            'subject' => $result->subject ? $result->subject->name : 'All Subjects',
            'institution' => $result->institution ? $result->institution->name : 'Unknown',
            'date' => $result->created_at->format('M d, Y'),
            'time' => $result->created_at->format('h:i A'),
            'created_at' => $result->created_at->toISOString(),
        ];
    });


            // Calculate achievements
            $achievements = $this->calculateAchievements($userId, $stats);

            // Get performance analytics
            $analytics = $this->getPerformanceAnalytics($userId);

            return response()->json([
                'success' => true,
                'stats' => $stats,
                'activity' => $recentActivity,
                'achievements' => $achievements,
                'analytics' => $analytics
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching user stats: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch user statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    private function calculateStudyStreak($userId)
    {
        // Get dates when user took quizzes
        $quizDates = QuizSession::where('user_id', $userId)
            ->select(DB::raw('DATE(created_at) as quiz_date'))
            ->distinct()
            ->orderBy('quiz_date', 'desc')
            ->pluck('quiz_date')
            ->toArray();
    
        if (empty($quizDates)) {
            return 0;
        }
    
        $streak = 1; // Start with 1 since we have at least one quiz date
        $currentDate = Carbon::today();
    
        // Check if user studied today or yesterday to start counting
        $lastQuizDate = Carbon::parse($quizDates[0]);
        $daysDiff = $currentDate->diffInDays($lastQuizDate);
    
        // If last quiz was more than 1 day ago, streak is broken
        if ($daysDiff > 1) {
            return 0;
        }
    
        // If last quiz was today, include it in streak, otherwise start from yesterday
        if ($daysDiff === 0) {
            $streak = 1;
        } else if ($daysDiff === 1) {
            $streak = 1;
        }
    
        // Count consecutive days backwards from the most recent quiz
        for ($i = 0; $i < count($quizDates) - 1; $i++) {
            $currentQuizDate = Carbon::parse($quizDates[$i]);
            $nextQuizDate = Carbon::parse($quizDates[$i + 1]);
            
            $diff = $currentQuizDate->diffInDays($nextQuizDate);
            
            if ($diff === 1) {
                $streak++;
            } else {
                break; // Streak broken, stop counting
            }
        }
    
        return $streak;
    }

    private function calculateAchievements($userId, $stats)
    {
        $achievements = [];

        // First Quiz Achievement
        if (QuizSession::where('user_id', $userId)->exists()) {
            $achievements[] = [
                'id' => 'first_quiz',
                'title' => 'First Steps',
                'description' => 'Completed your first quiz',
                'icon' => '🎯',
                'earned' => true,
                'earned_at' => QuizSession::where('user_id', $userId)->oldest()->first()->created_at->format('M d, Y')
            ];
        }

        // High Score Achievement
        $bestScore = QuizSession::where('user_id', $userId)->max('score');
        if ($bestScore >= 80) {
            $achievements[] = [
                'id' => 'high_scorer',
                'title' => 'High Scorer',
                'description' => 'Achieved a score of 80% or higher',
                'icon' => '🏆',
                'earned' => true,
                'earned_at' => QuizSession::where('user_id', $userId)->where('score', '>=', 80)->latest()->first()->created_at->format('M d, Y')
            ];
        }

        // Study Streak Achievement
        if ($stats['studyStreak'] >= 7) {
            $achievements[] = [
                'id' => 'week_warrior',
                'title' => 'Week Warrior',
                'description' => 'Maintained a 7-day study streak',
                'icon' => '🔥',
                'earned' => true,
                'earned_at' => Carbon::now()->format('M d, Y')
            ];
        }

        // Quiz Master Achievement
        if ($stats['questionsAnswered'] >= 100) {
            $achievements[] = [
                'id' => 'quiz_master',
                'title' => 'Quiz Master',
                'description' => 'Answered 100+ questions',
                'icon' => '📚',
                'earned' => true,
                'earned_at' => Carbon::now()->format('M d, Y')
            ];
        }

        // Content Creator Achievement
        if ($stats['questionsUploaded'] >= 10) {
            $achievements[] = [
                'id' => 'content_creator',
                'title' => 'Content Creator',
                'description' => 'Uploaded 10+ questions',
                'icon' => '✍️',
                'earned' => true,
                'earned_at' => Carbon::now()->format('M d, Y')
            ];
        }

        // Add some unearned achievements for motivation
        if ($stats['studyStreak'] < 30) {
            $achievements[] = [
                'id' => 'monthly_master',
                'title' => 'Monthly Master',
                'description' => 'Maintain a 30-day study streak',
                'icon' => '🌟',
                'earned' => false,
                'progress' => $stats['studyStreak'],
                'target' => 30
            ];
        }

        if ($bestScore < 95) {
            $achievements[] = [
                'id' => 'perfectionist',
                'title' => 'Perfectionist',
                'description' => 'Achieve a 95% score or higher',
                'icon' => '💎',
                'earned' => false,
                'progress' => $bestScore,
                'target' => 95
            ];
        }

        return $achievements;
    }

    private function getPerformanceAnalytics($userId)
    {
        // Get performance over last 30 days
        $thirtyDaysAgo = Carbon::now()->subDays(30);
        
        $dailyPerformance = QuizResult::where('user_id', $userId)
            ->where('created_at', '>=', $thirtyDaysAgo)
            ->select(
                DB::raw('DATE(created_at) as date'),
                DB::raw('AVG(CASE WHEN total_questions > 0 THEN (correct_answers / total_questions) * 100 ELSE 0 END) as avg_score'),
                DB::raw('COUNT(*) as quizzes_taken')
            )
            ->groupBy('date')
            ->orderBy('date')
            ->get();
    
        // Get subject-wise performance
        $subjectPerformance = QuizResult::where('user_id', $userId)
            ->select(
                'quiz_type',
                DB::raw('AVG(CASE WHEN total_questions > 0 THEN (correct_answers / total_questions) * 100 ELSE 0 END) as avg_score'),
                DB::raw('COUNT(*) as attempts')
            )
            ->groupBy('quiz_type')
            ->get();
    
        return [
            'daily_performance' => $dailyPerformance,
            'subject_performance' => $subjectPerformance,
            'total_time_studied' => QuizResult::where('user_id', $userId)->sum('time_spent'),
            'average_score' => QuizResult::where('user_id', $userId)
                ->where('total_questions', '>', 0) // Fixed: Use where instead of whereColumn
                ->avg(DB::raw('(correct_answers / total_questions) * 100')) ?? 0,
            'best_subject' => $subjectPerformance->sortByDesc('avg_score')->first()?->quiz_type ?? 'N/A',
            'improvement_rate' => $this->calculateImprovementRate($userId)
        ];
    }

    private function calculateImprovementRate($userId)
    {
        $firstHalfAvg = QuizResult::where('user_id', $userId)
            ->oldest()
            ->limit(10)
            ->avg(DB::raw('CASE WHEN total_questions > 0 THEN (correct_answers / total_questions) * 100 ELSE 0 END')) ?? 0;

        $secondHalfAvg = QuizResult::where('user_id', $userId)
            ->latest()
            ->limit(10)
            ->avg(DB::raw('CASE WHEN total_questions > 0 THEN (correct_answers / total_questions) * 100 ELSE 0 END')) ?? 0;

        return round($secondHalfAvg - $firstHalfAvg, 1);
    }
    public function logout(Request $request)
    {
        // Revoke the current user's token
        $request->user()->currentAccessToken()->delete();


        return response()->json([
            'message' => 'Successfully logged out'
        ], 200);
    }
    
}

