<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PersonalQuestion;
use Carbon\Carbon;

class FlashcardController extends Controller
{
    /**
     * Get cards due for review today
     */
    public function getDueCards(Request $request)
    {
        $userId = auth()->id();
        $now = Carbon::now();

        $dueQuestions = PersonalQuestion::where('user_id', $userId)
            ->where('due_date', '<=', $now)
            // ->where('is_active', true)
            ->orderBy('due_date', 'asc')
            ->get();

        return response()->json([
            'cards' => $this->transformToFlashcards($dueQuestions),
            'total' => $dueQuestions->count(),
            'deck_name' => "Today's Reviews"
        ]);
    }

    /**
     * Get cards by course (subject)
     */
    public function getCardsByCourse(Request $request, $courseId)
    {
        $userId = auth()->id();

        $questions = PersonalQuestion::where('user_id', $userId)
            ->where('course_id', $courseId)
            // ->where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->get();

        $course = \App\Models\Course::find($courseId);

        return response()->json([
            'cards' => $this->transformToFlashcards($questions),
            'total' => $questions->count(),
            'deck_name' => $course ? $course->name : 'Course Cards'
        ]);
    }

    /**
     * Get all cards
     */
    public function getAllCards(Request $request)
    {
        $userId = auth()->id();

        $questions = PersonalQuestion::where('user_id', $userId)
            // ->where('is_active', true)
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'cards' => $this->transformToFlashcards($questions),
            'total' => $questions->count(),
            'deck_name' => 'All Cards'
        ]);
    }

    /**
     * Get recently missed cards
     */
    public function getMistakes(Request $request)
    {
        $userId = auth()->id();

        // Cards marked wrong in the last 7 days
        $mistakes = PersonalQuestion::where('user_id', $userId)
            ->where('last_reviewed', '>=', Carbon::now()->subDays(7))
            ->where('incorrect_count', '>', 0)
            // ->where('is_active', true)
            ->orderBy('last_reviewed', 'desc')
            ->get();
            $mistakes_count = PersonalQuestion::where('user_id', $userId)
            ->where('last_reviewed', '>=', Carbon::now()->subDays(7))
            ->where('incorrect_count', '>', 0)
            ->count();
        return response()->json([
            'cards' => $this->transformToFlashcards($mistakes),
            'total' => $mistakes_count,
            'deck_name' => 'Recent Mistakes'
        ]);
    }

    /**
     * Submit a card review
     */
    public function submitReview(Request $request, $questionId)
    {
        $request->validate([
            'rating' => 'required|in:again,hard,good,easy',
            'time_taken_seconds' => 'integer|min:0'
        ]);
        $user = auth()->user();
        $userId = auth()->id();
        $rating = $request->rating;
        $timeTaken = $request->time_taken_seconds ?? 0;

        $question = PersonalQuestion::where('id', $questionId)
            ->where('user_id', $userId)
            ->firstOrFail();

        // Store previous state
        $previousState = [
            'ease_factor' => $question->ease_factor ?? 2.5,
            'interval_days' => $question->interval_days ?? 0,
            'repetitions' => $question->repetitions ?? 0,
        ];

        // Calculate new state using SRS algorithm
        $newState = $this->calculateNextReview(
            $rating,
            $previousState['ease_factor'],
            $previousState['interval_days'],
            $previousState['repetitions']
        );

        // Update question with new SRS values
        $question->ease_factor = $newState['ease_factor'];
        $question->interval_days = $newState['interval_days'];
        $question->repetitions = $newState['repetitions'];
        $question->due_date = $newState['due_date'];
        $question->last_reviewed = Carbon::now();
        $question->review_count = ($question->review_count ?? 0) + 1;
        $question->status = $newState['status'];

        if (in_array($rating, ['good', 'easy'])) {
            $question->correct_count = ($question->correct_count ?? 0) + 1;
        } elseif ($rating === 'again') {
            $question->incorrect_count = ($question->incorrect_count ?? 0) + 1;
        }

        $question->save();
        // === UPDATE STREAK LOGIC ===
    $today = Carbon::today();

    if ($user->last_study_date) {
        $lastStudy = Carbon::parse($user->last_study_date);

        if ($lastStudy->isToday()) {

            $newStreak = $user->streak;
        } elseif ($lastStudy->isYesterday()) {
           
            $newStreak = $user->streak + 1;
        } else {
            
            $newStreak = 1;
        }
    } else {
      
        $newStreak = 1;
    }

   
    $user->streak = $newStreak;
    $user->last_study_date = $today;
    $user->save();

        // Log review history
        \DB::table('question_review_history')->insert([
            'user_id' => $userId,
            'question_id' => $questionId,
            'rating' => $rating,
            'time_taken_seconds' => $timeTaken,
            'previous_interval' => $previousState['interval_days'],
            'previous_ease_factor' => $previousState['ease_factor'],
            'new_interval' => $newState['interval_days'],
            'new_ease_factor' => $newState['ease_factor'],
            'new_due_date' => $newState['due_date'],
            'reviewed_at' => Carbon::now()
        ]);

        return response()->json([
            'success' => true,
            'next_review_date' => $newState['due_date']->toISOString(),
            'interval_days' => $newState['interval_days'],
            'status' => $newState['status'],
            'streak' => $newStreak,
            'message' => "Card scheduled for review in {$newState['interval_days']} days"
        ]);
    }

    /**
     * SM-2 Algorithm Implementation
     */
    private function calculateNextReview($rating, $easeFactor, $intervalDays, $repetitions)
    {
        $newEaseFactor = $easeFactor;
        $newInterval = $intervalDays;
        $newRepetitions = $repetitions;
        $status = 'learning';

        switch ($rating) {
            case 'again':
                $newEaseFactor = max(1.3, $easeFactor - 0.2);
                $newInterval = 1;
                $newRepetitions = 0;
                $status = 'relearning';
                break;

            case 'hard':
                $newEaseFactor = max(1.3, $easeFactor - 0.15);
                $newInterval = max(1, round($intervalDays * 1.2));
                $newRepetitions = $repetitions + 1;
                $status = $newRepetitions >= 2 ? 'review' : 'learning';
                break;

            case 'good':
                if ($repetitions == 0) {
                    $newInterval = 1;
                } elseif ($repetitions == 1) {
                    $newInterval = 6;
                } else {
                    $newInterval = round($intervalDays * $easeFactor);
                }
                $newRepetitions = $repetitions + 1;
                $status = $newRepetitions >= 2 ? 'review' : 'learning';
                break;

            case 'easy':
                $newEaseFactor = min(2.5, $easeFactor + 0.15);
                if ($repetitions == 0) {
                    $newInterval = 4;
                } elseif ($repetitions == 1) {
                    $newInterval = 10;
                } else {
                    $newInterval = round($intervalDays * $easeFactor * 1.3);
                }
                $newRepetitions = $repetitions + 1;
                $status = 'review';
                break;
        }

        return [
            'ease_factor' => round($newEaseFactor, 2),
            'interval_days' => $newInterval,
            'repetitions' => $newRepetitions,
            'status' => $status,
            'due_date' => Carbon::now()->addDays($newInterval)
        ];
    }

    /**
     * Transform questions to flashcard format
     */
    private function transformToFlashcards($questions)
    {
        return $questions->map(function ($question) {
            $optionsArray = json_decode($question->options, true) ?? [];
            
            // Format options as ["A) ...", "B) ...", "C) ...", "D) ..."]
            $formattedOptions = [];
            foreach ($optionsArray as $index => $option) {
                $letter = chr(65 + $index); // A, B, C, D
                $formattedOptions[] = "{$letter}) {$option}";
            }

            // Get correct answer text
            $answerIndex = (int)$question->answer;
            $correctAnswer = $optionsArray[$answerIndex] ?? '';
            $answerLetter = chr(65 + $answerIndex);

            return [
                'id' => $question->id,
                'front' => $question->question,
                'back' => "{$answerLetter}) {$correctAnswer}",
                'explanation' => $question->explanation,
                'options' => $formattedOptions,
                'image' => null, // Add if you have images
            ];
        });
    }

    /**
     * Get user flashcard statistics
     */
    public function getStats(Request $request)
    {
        $user = auth()->user();
        $userId = auth()->id();
        $now = Carbon::now();

        $totalCards = PersonalQuestion::where('user_id', $userId)->count();
        $dueToday = PersonalQuestion::where('user_id', $userId)
            ->where('due_date', '<=', $now)
            ->count();
        $newCards = PersonalQuestion::where('user_id', $userId)
            ->where('status', 'new')
            ->count();
            $mistakes = PersonalQuestion::where('user_id', $userId)
            ->where('last_reviewed', '>=', Carbon::now()->subDays(7))
            ->where('incorrect_count', '>', 0)
            // ->where('is_active', true)
            ->count();

        // Calculate accuracy
        $totalReviews = \DB::table('question_review_history')
            ->where('user_id', $userId)
            ->count();
        $correctReviews = \DB::table('question_review_history')
            ->where('user_id', $userId)
            ->whereIn('rating', ['good', 'easy'])
            ->count();

        $accuracy = $totalReviews > 0 ? round(($correctReviews / $totalReviews) * 100, 1) : 0;

        return response()->json([
            'dueToday' => $dueToday,
            'totalCards' => $totalCards,
            'newToday' => $newCards,
            'accuracy' => $accuracy,
            'mistakes' => $mistakes,
            'streak' => $user->streak ?? 0
        ]);
    }
}
