<?php

namespace App\Http\Controllers;

use App\Services\MedicalQuestionGeneratorService;
use App\Models\Questions;
use App\Models\Subjects;
use App\Models\Subtopics;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class MedicalQuestionGeneratorController extends Controller
{
    protected $generator;

    public function __construct(MedicalQuestionGeneratorService $generator)
    {
        $this->generator = $generator;
    }

    /**
     * Generate questions automatically (with optional count, defaults to 10)
     */
    public function generate(Request $request)
    {
        $validated = $request->validate([
            'count'         => 'sometimes|integer|min:1|max:50',
            'subject_id'    => 'required|integer|exists:subjects,id',
            'subtopic_id'   => 'required|integer|exists:subtopics,id',
            'uploaded_by'   => 'required|integer',
            'school_id'     => 'required|integer',
        ]);

        try {
            $subject  = Subjects::findOrFail($validated['subject_id']);
            $subtopic = Subtopics::findOrFail($validated['subtopic_id']);

            if ($subtopic->subject_id != $subject->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Subtopic does not belong to the specified subject',
                ], 400);
            }

            $count = $validated['count'] ?? 20;

            $params = [
                'count'         => $count,
                'subject_id'    => $subject->id,
                'subtopic_id'   => $subtopic->id,
                'subject_name'  => $subject->name,
                'subtopic_name' => $subtopic->name,
                'uploaded_by'   => $validated['uploaded_by'],
                'school_id'     => $validated['school_id'],
            ];

            $result = $this->generator->generateQuestions($params);

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to generate questions',
                    'error'   => $result['error'] ?? 'Unknown error',
                ], 500);
            }

            return response()->json([
                'success'  => true,
                'message'  => "{$result['count']} questions generated and saved successfully",
                'count'    => $result['count'],
                'subject'  => $subject->name,
                'subtopic' => $subtopic->name,
                'questions'=> $result['questions'],
            ]);

        } catch (\Exception $e) {
            Log::error('Question generation failed', [
                'error'   => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
                'request' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred during question generation',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * AUTO-GENERATE: Pick random subject + random subtopic and generate questions instantly
     */
    public function autoGenerate(Request $request)
    {
        $validated = $request->validate([
            'count'       => 'sometimes|integer|min:1|max:30',
            'uploaded_by' => 'sometimes|integer',
            'school_id'   => 'sometimes|integer',
        ]);
    
        try {
            // Pick random subject WHERE auto = 1
            $subject = Subjects::where('auto', 1)
                ->inRandomOrder()
                ->first();
    
            if (!$subject) {
                return response()->json([
                    'success' => false,
                    'message' => 'No subjects with auto-generation enabled found in the database',
                ], 404);
            }
    
            // Pick random subtopic from that subject
            $subtopic = $subject->subtopics()->inRandomOrder()->first();
    
            if (!$subtopic) {
                return response()->json([
                    'success' => false,
                    'message' => "No subtopics found for subject: {$subject->name}",
                ], 404);
            }
    
            $count = $validated['count'] ?? 10;
            $uploadedBy = $validated['uploaded_by'] ?? 1;
            $schoolId = $validated['school_id'] ?? 1;
    
            $params = [
                'count'         => $count,
                'subject_id'    => $subject->id,
                'subtopic_id'   => $subtopic->id,
                'subject_name'  => $subject->name,
                'subtopic_name' => $subtopic->name,
                'uploaded_by'   => $uploadedBy,
                'school_id'     => $schoolId,
            ];
    
            $result = $this->generator->generateQuestions($params);
    
            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to generate questions',
                    'error'   => $result['error'] ?? 'Unknown error',
                ], 500);
            }
    
            return response()->json([
                'success'   => true,
                'message'   => "{$result['count']} questions generated for auto-enabled topic!",
                'count'     => $result['count'],
                'subject'   => $subject->name,
                'subtopic'  => $subtopic->name,
                'questions' => $result['questions'],
            ]);
    
        } catch (\Exception $e) {
            Log::error('Auto generation failed', [
                'error'   => $e->getMessage(),
                'request' => $request->all(),
            ]);
    
            return response()->json([
                'success' => false,
                'message' => 'An error occurred during auto generation',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generate questions in smaller batches
     */
    public function generateBatch(Request $request)
    {
        $validated = $request->validate([
            'batch_size'   => 'sometimes|integer|min:1|max:15',
            'subject_id'   => 'required|integer|exists:subjects,id',
            'subtopic_id'  => 'required|integer|exists:subtopics,id',
            'uploaded_by'  => 'required|integer',
            'school_id'    => 'required|integer',
        ]);

        try {
            $subject  = Subjects::findOrFail($validated['subject_id']);
            $subtopic = Subtopics::findOrFail($validated['subtopic_id']);

            if ($subtopic->subject_id != $subject->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Subtopic does not belong to the specified subject',
                ], 400);
            }

            $batchSize = $validated['batch_size'] ?? 10;

            $params = [
                'count'         => $batchSize,
                'subject_id'    => $subject->id,
                'subtopic_id'   => $subtopic->id,
                'subject_name'  => $subject->name,
                'subtopic_name' => $subtopic->name,
                'uploaded_by'   => $validated['uploaded_by'],
                'school_id'     => $validated['school_id'],
            ];

            $result = $this->generator->generateBatch($batchSize, $params);

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to generate questions',
                    'error'   => $result['error'] ?? null,
                ], 500);
            }

            return response()->json([
                'success'  => true,
                'message'  => "Batch of {$result['count']} questions generated successfully",
                'count'    => $result['count'],
                'subject'  => $subject->name,
                'subtopic' => $subtopic->name,
                'questions'=> $result['questions'],
            ]);

        } catch (\Exception $e) {
            Log::error('Batch generation failed', [
                'error'   => $e->getMessage(),
                'request' => $request->all(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Batch generation failed',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    // ... (rest of your methods remain unchanged)

    public function getSubjects()
    {
        $subjects = Subjects::select('id', 'name', 'description')
            ->orderBy('name')
            ->get();

        return response()->json([
            'success'  => true,
            'subjects' => $subjects,
        ]);
    }

    public function getSubtopics(Request $request)
    {
        $validated = $request->validate([
            'subject_id' => 'required|integer|exists:subjects,id',
        ]);

        $subtopics = Subtopics::where('subject_id', $validated['subject_id'])
            ->select('id', 'name', 'subject_id')
            ->orderBy('name')
            ->get();

        return response()->json([
            'success'   => true,
            'subtopics' => $subtopics,
        ]);
    }

    public function status(Request $request)
    {
        $subjectId  = $request->input('subject_id');
        $subtopicId = $request->input('subtopic_id');

        $query = Questions::query();

        if ($subjectId) {
            $query->where('subject_id', $subjectId);
        }
        if ($subtopicId) {
            $query->where('subtopic_id', $subtopicId);
        }

        $totalQuestions = $query->count();

        $difficultyBreakdown = $query->clone()
            ->select('difficulty')
            ->selectRaw('count(*) as count')
            ->groupBy('difficulty')
            ->get();

        $recentQuestions = Questions::query()
            ->when($subjectId, fn($q) => $q->where('subject_id', $subjectId))
            ->when($subtopicId, fn($q) => $q->where('subtopic_id', $subtopicId))
            ->with(['subject:id,name', 'subtopic:id,name'])
            ->latest()
            ->take(10)
            ->get(['id', 'question', 'difficulty', 'subject_id', 'subtopic_id', 'created_at']);

        return response()->json([
            'success'             => true,
            'total_questions'     => $totalQuestions,
            'difficulty_breakdown'=> $difficultyBreakdown,
            'recent_questions'    => $recentQuestions,
        ]);
    }

    public function generateForSubject(Request $request)
    {
        $validated = $request->validate([
            'subject_id'            => 'required|integer|exists:subjects,id',
            'questions_per_subtopic'=> 'sometimes|integer|min:1|max:20',
            'uploaded_by'           => 'required|integer',
            'school_id'             => 'required|integer',
        ]);

        try {
            $subject = Subjects::findOrFail($validated['subject_id']);
            $subtopics = Subtopics::where('subject_id', $validated['subject_id'])->get();

            $results = [];
            $totalGenerated = 0;
            $perSubtopic = $validated['questions_per_subtopic'] ?? 10;

            foreach ($subtopics as $subtopic) {
                $params = [
                    'count'         => $perSubtopic,
                    'subject_id'    => $subject->id,
                    'subtopic_id'   => $subtopic->id,
                    'subject_name'  => $subject->name,
                    'subtopic_name' => $subtopic->name,
                    'uploaded_by'   => $validated['uploaded_by'],
                    'school_id'     => $validated['school_id'],
                ];

                $result = $this->generator->generateQuestions($params);

                if ($result['success']) {
                    $results[] = [
                        'subtopic' => $subtopic->name,
                        'count'    => $result['count'],
                    ];
                    $totalGenerated += $result['count'];
                }

                sleep(2);
            }

            return response()->json([
                'success'         => true,
                'message'         => "Generated {$totalGenerated} questions across " . count($results) . " subtopics",
                'total_generated' => $totalGenerated,
                'results'         => $results,
            ]);

        } catch (\Exception $e) {
            Log::error('generateForSubject failed', ['error' => $e->getMessage()]);

            return response()->json([
                'success' => false,
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function preview(Request $request)
    {
        $validated = $request->validate([
            'subject_id' => 'required|integer|exists:subjects,id',
            'subtopic_id'=> 'required|integer|exists:subtopics,id',
            'count'      => 'sometimes|integer|min:1|max:5',
        ]);

        $subject  = Subjects::findOrFail($validated['subject_id']);
        $subtopic = Subtopics::findOrFail($validated['subtopic_id']);

        if ($subtopic->subject_id != $subject->id) {
            return response()->json([
                'success' => false,
                'message' => 'Subtopic does not belong to the specified subject',
            ], 400);
        }

        return response()->json([
            'success'  => true,
            'preview'  => true,
            'subject'  => $subject->name,
            'subtopic' => $subtopic->name,
            'count'    => $validated['count'] ?? 3,
            'note'     => 'This is a preview. Use /generate or /auto-generate to save questions.',
        ]);
    }

    public function delete(Request $request)
    {
        $validated = $request->validate([
            'question_ids'   => 'required|array',
            'question_ids.*' => 'integer|exists:questions,id',
        ]);

        try {
            $deleted = Questions::whereIn('id', $validated['question_ids'])->delete();

            return response()->json([
                'success'       => true,
                'message'       => "{$deleted} questions deleted successfully",
                'deleted_count' => $deleted,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error'   => $e->getMessage(),
            ], 500);
        }
    }
}