<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Smalot\PdfParser\Parser;
use Auth;
use App\Models\PersonalQuestion;
use App\Models\Course;

class QuestionGeneratorController extends Controller
{
    protected $aiProvider = 'gemini'; // Default provider

    public function ai_generateQuestions(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:pdf|max:' . env('MAX_FILE_SIZE', 51200),
            'question_count' => 'required|integer|min:1|max:50',
            'course_name' => 'required|string|max:255',
            'subject_id' => 'nullable|integer|exists:subjects,id',
            'subtopic_id' => 'nullable|integer|exists:sub_topics,id',
            'ai_provider' => 'sometimes|string|in:deepseek,gemini',
        ]);
    
        set_time_limit(300);
        $courseName = $request->input('course_name');
    
        $allQuestions = []; 
    
        try {
            // Store and parse PDF
            $file = $request->file('file');
            $path = $file->store('temp');
            $fullPath = Storage::path($path);
    
            try {
                $parser = new Parser();
                $pdf = $parser->parseFile($fullPath);
                $extractedText = $pdf->getText();
                Log::info('PDF parsed successfully', ['length' => strlen($extractedText)]);
            } catch (\Exception $e) {
                Log::error('PDF extraction failed: ' . $e->getMessage());
                throw new \Exception('Failed to extract text from PDF. Ensure it contains readable text (not scanned/image-only).');
            } finally {
                Storage::delete($path);
            }
    
            $extractedText = trim($extractedText);
            if (empty($extractedText)) {
                return response()->json(['error' => ['message' => 'No text found in PDF.', 'code' => 'EMPTY_TEXT']], 422);
            }
    
            $questionCount = (int) $request->input('question_count');
            if (strlen($extractedText) < $questionCount * 100) {
                return response()->json([
                    'error' => ['message' => "Text too short to generate $questionCount questions.", 'code' => 'INSUFFICIENT_CONTENT']
                ], 422);
            }
    
            // Split into chunks
            $chunkSize = 300000;
            $chunks = $this->splitText($extractedText, $chunkSize);
            $questionsPerChunk = ceil($questionCount / max(1, count($chunks)));
    
            $maxRetries = 3;
            $retryCount = 0;
            $networkErrorOccurred = false;
    
            while (count($allQuestions) < $questionCount && $retryCount < $maxRetries) {
                foreach ($chunks as $index => $chunk) {
                    $needed = $questionCount - count($allQuestions);
                    $chunkQuestionCount = min($questionsPerChunk, $needed);
                    if ($chunkQuestionCount <= 0) break;
    
                    $chunk = str_replace(['"', "\n", "\r"], ['\"', ' ', ' '], $chunk);
                    $prompt = $this->buildPrompt($chunkQuestionCount, $chunk);
    
                    try {
                        $questions = $this->aiProvider === 'gemini'
                            ? $this->callGeminiApi($prompt)
                            : $this->callDeepSeekApi($prompt);
    
                        $allQuestions = array_merge($allQuestions, $questions);
                        Log::info("Chunk $index: Generated " . count($questions) . " questions");
                    } catch (\GuzzleHttp\Exception\ConnectException | \GuzzleHttp\Exception\RequestException $e) {
                        // Network-related errors (timeout, no connection, DNS, etc.)
                        $networkErrorOccurred = true;
                        Log::warning("Network/AI timeout on chunk $index (retry $retryCount): " . $e->getMessage());
                    } catch (\Exception $e) {
                        Log::warning("Failed chunk $index on retry $retryCount: " . $e->getMessage());
                    }
    
                    if (count($allQuestions) >= $questionCount) break;
                }
                $retryCount++;
            }
    
            $allQuestions = array_slice($allQuestions, 0, $questionCount);
    
            // === Critical Check: Did we fail due to network or insufficient generation? ===
            if (count($allQuestions) === 0) {
                if ($networkErrorOccurred) {
                    return response()->json([
                        'error' => [
                            'message' => 'Connection issue or AI service unavailable. Please check your internet connection and try again.',
                            'code' => 'NETWORK_ERROR'
                        ]
                    ], 502);
                }
    
                return response()->json([
                    'error' => [
                        'message' => 'Failed to generate any questions. The document may not be suitable for AI processing.',
                        'code' => 'GENERATION_FAILED'
                    ]
                ], 422);
            }
    
            if (count($allQuestions) < $questionCount * 0.7) {
                if ($networkErrorOccurred) {
                    return response()->json([
                        'error' => [
                            'message' => 'Unstable connection prevented generating all questions. Please try again with a stable network.',
                            'code' => 'PARTIAL_NETWORK_FAILURE'
                        ]
                    ], 502);
                }
    
                return response()->json([
                    'error' => [
                        'message' => 'Could not generate enough quality questions from this document. Try a more detailed PDF.',
                        'code' => 'INSUFFICIENT_QUESTIONS'
                    ]
                ], 422);
            }
    
            // === ONLY NOW: Create course and save questions (safe zone) ===
            $course = Course::create([
                'name' => $courseName,
                'user_id' => Auth::id(),
                'subject_id' => $request->subject_id,
                'subtopic_id' => $request->subtopic_id
            ]);
    
            $savedQuestions = $this->saveQuestionsToDatabase($allQuestions, $course->id, $request->subject_id, $request->subtopic_id);
    
            return response()->json([
                'questions' => $savedQuestions,
                'message' => 'Questions generated successfully using ' . ucfirst($this->aiProvider),
                'provider' => $this->aiProvider,
                'generated_count' => count($savedQuestions)
            ], 200);
    
        } catch (\Exception $e) {
            Log::error('Question generation failed: ' . $e->getMessage());
            return response()->json([
                'error' => ['message' => $e->getMessage(), 'code' => 'GENERAL_ERROR']
            ], 500);
        }
    }

    private function buildPrompt($count, $content)
{
    return "You are a senior medical educator and board-style exam question writer.
Generate EXACTLY $count high-quality multiple-choice questions suitable for medical students preparing for MB (MBBS/MBChB) examinations.

QUESTION STANDARDS:
- Mix of question types:
  • Direct knowledge-based questions
  • Clinical scenario (vignette-based) questions
  • Tricky questions testing common misconceptions
  • 'EXCEPT' or 'NOT' type questions (clearly stated)
- Questions must test understanding, application, and clinical reasoning
- Clinical questions should include brief patient history, symptoms, or findings where appropriate
- Avoid ambiguous wording; only ONE option must be clearly correct
- Use medically accurate terminology at undergraduate level

OPTION RULES:
- EXACTLY 4 options per question
- Options should be plausible and exam-standard distractors
- Do NOT use options like 'All of the above' or 'None of the above'

OUTPUT RULES (STRICT):
- Output ONLY a valid JSON array
- NO markdown
- NO code blocks
- NO numbering
- NO extra text
- Start directly with [ and end with ]
- correct_answer must be an integer between 0–3 (index of the correct option)
- explanation must be brief, clear, and clinically relevant (1–2 sentences)

JSON FORMAT (FOLLOW EXACTLY):
[
  {
    \"question\": \"Question text?\",
    \"options\": [\"Option A\", \"Option B\", \"Option C\", \"Option D\"],
    \"correct_answer\": 0,
    \"explanation\": \"Brief medical explanation\"
  }
]

CONTENT TO USE:
$content

Generate exactly $count questions now.";
}

    private function callGeminiApi($prompt)
    {
        $client = new Client();
        $apiKey = env('GEMINI_API_KEY');
        $endpoint = env('GEMINI_API_ENDPOINT', 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent');

        if (!$apiKey) throw new \Exception('GEMINI_API_KEY not set');

        $response = $client->post($endpoint . '?key=' . $apiKey, [
            'json' => [
                'contents' => [['parts' => [['text' => $prompt]]]],
                'generationConfig' => [
                    'temperature' => 0.3,
                    'maxOutputTokens' => 8192,
                    'responseMimeType' => 'application/json',
                    'responseSchema' => [
                        'type' => 'array',
                        'items' => [
                            'type' => 'object',
                            'properties' => [
                                'question' => ['type' => 'string'],
                                'options' => ['type' => 'array', 'items' => ['type' => 'string'], 'minItems' => 4, 'maxItems' => 4],
                                'correct_answer' => ['type' => 'integer', 'minimum' => 0, 'maximum' => 3],
                                'explanation' => ['type' => 'string']
                            ],
                            'required' => ['question', 'options', 'correct_answer', 'explanation']
                        ]
                    ]
                ]
            ],
            'timeout' => 90
        ]);

        $data = json_decode($response->getBody(), true);
        $text = $data['candidates'][0]['content']['parts'][0]['text'] ?? '';

        return $this->parseAndValidateQuestions($text, 'Gemini');
    }

    private function callDeepSeekApi($prompt)
    {
        $client = new Client();
        $token = env('HF_TOKEN');
        $model = 'deepseek-ai/DeepSeek-V3.2:novita';

        if (!$token) throw new \Exception('HUGGINGFACE_API_TOKEN not set');

        $response = $client->post('https://router.huggingface.co/v1/chat/completions', [
            'headers' => ['Authorization' => 'Bearer ' . $token],
            'json' => [
                'model' => $model,
                'messages' => [['role' => 'user', 'content' => $prompt]],
                'max_tokens' => 4096,
                'temperature' => 0.3,
                'response_format' => ['type' => 'json_object']
            ],
            'timeout' => 120
        ]);

        $data = json_decode($response->getBody(), true);
        $content = $data['choices'][0]['message']['content'] ?? '';

        if (empty(trim($content))) {
            throw new \Exception('Empty response from DeepSeek');
        }

        Log::info('DeepSeek raw response preview', ['preview' => substr($content, 0, 500)]);

        return $this->parseAndValidateQuestions($content, 'DeepSeek');
    }

    private function parseAndValidateQuestions($content, $provider)
    {
        $content = trim($content);

        // Clean markdown and extract JSON array
        $content = preg_replace('/^```json\s*/i', '', $content);
        $content = preg_replace('/\s*```$/i', '', $content);
        $content = preg_replace('/^[\s\S]*?(\[[\s\S]*\])[\s\S]*$/', '$1', $content); // Extract [ ... ]
        $content = preg_replace('/,\s*([\]}])/', '$1', $content); // Remove trailing commas

        $questions = json_decode($content, true);

        if (json_last_error() !== JSON_ERROR_NONE || !is_array($questions)) {
            Log::warning("$provider JSON parse failed, attempting fallback recovery");
            $questions = $this->recoverQuestionsFromText($content);
        }

        if (empty($questions)) {
            throw new \Exception("No valid questions parsed from $provider response");
        }

        // Final validation
        $validQuestions = [];
        foreach ($questions as $q) {
            if (
                isset($q['question'], $q['options'], $q['correct_answer'], $q['explanation']) &&
                is_array($q['options']) && count($q['options']) === 4 &&
                is_int($q['correct_answer']) && $q['correct_answer'] >= 0 && $q['correct_answer'] <= 3
            ) {
                $validQuestions[] = $q;
            }
        }

        Log::info("Parsed " . count($validQuestions) . " valid questions from $provider");
        return $validQuestions;
    }

    private function recoverQuestionsFromText($text)
    {
        $questions = [];
        preg_match_all('/\{[^{}]*"question"[^{}]*"options"[^{}]*"correct_answer"[^{}]*"explanation"[^{}]*\}/s', $text, $matches);

        foreach ($matches[0] as $match) {
            $decoded = json_decode($match, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $questions[] = $decoded;
            }
        }

        return $questions;
    }

    private function saveQuestionsToDatabase($questions, $courseId, $subjectId, $subtopicId)
    {
        $saved = [];
        $userId = Auth::id();

        foreach ($questions as $q) {
            try {
                $pq = PersonalQuestion::create([
                    'course_id' => $courseId,
                    'user_id' => $userId,
                    'subject_id' => $subjectId,
                    'subtopic_id' => $subtopicId,
                    'question' => $q['question'],
                    'answer' => $q['correct_answer'],
                    'explanation' => $q['explanation'],
                    'options' => json_encode($q['options']),
                ]);
                $q['id'] = $pq->id;
                $saved[] = $q;
            } catch (\Exception $e) {
                Log::error('Failed to save question: ' . $e->getMessage());
            }
        }

        return $saved;
    }

    private function splitText($text, $chunkSize = 300000)
    {
        $chunks = [];
        while (strlen($text) > 0) {
            $chunk = substr($text, 0, $chunkSize);
            $lastBreak = max(strrpos($chunk, '.'), strrpos($chunk, "\n"), strrpos($chunk, ' '));
            if ($lastBreak > $chunkSize * 0.8) {
                $chunk = substr($chunk, 0, $lastBreak + 1);
            }
            $chunks[] = trim($chunk);
            $text = substr($text, strlen($chunk));
        }
        return $chunks;
    }

    public function user_questions(Request $request)
    {
        return PersonalQuestion::where('user_id', Auth::id())->get();
    }
}