<?php

namespace App\Services;

use App\Models\Questions;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class MedicalQuestionGeneratorService
{
    protected HuggingFaceService $huggingFace;
    protected string $model = 'deepseek-ai/DeepSeek-V3.2:novita';

    public function __construct(HuggingFaceService $huggingFace)
    {
        $this->huggingFace = $huggingFace;
    }

    /**
     * Generate and insert medical MCQs
     */
    public function generateQuestions(array $params): array
    {
        $count        = $params['count'] ?? 20;
        $subjectId    = $params['subject_id'];
        $subtopicId   = $params['subtopic_id'];
        $subjectName  = $params['subject_name'];
        $subtopicName = $params['subtopic_name'];
        $uploadedBy   = $params['uploaded_by'];
        $schoolId     = $params['school_id'];

        $prompt = $this->buildPrompt($count, $subtopicName, $subjectName);

        Log::info('Generating medical MCQs', [
            'model'    => $this->model,
            'count'    => $count,
            'subject'  => $subjectName,
            'subtopic' => $subtopicName,
        ]);

        // Calculate tokens: ~500 tokens per detailed medical MCQ + safety margin
        $tokensPerQuestion = 10000;
        $safetyMargin = 1200;
        $maxTokens = min(($count * $tokensPerQuestion) + $safetyMargin, 9000);
        
        Log::info('Token calculation', [
            'requested_count' => $count,
            'tokens_per_question' => $tokensPerQuestion,
            'max_tokens' => $maxTokens,
        ]);

        $response = $this->huggingFace->query(
            $this->model,
            $prompt,
            [
                'max_tokens'  => $maxTokens,
                'temperature' => 0.4,
                'top_p'       => 0.95,
            ]
        );

        if (isset($response['error'])) {
            return [
                'success' => false,
                'error'   => $response['error'],
                'count'   => 0,
            ];
        }

        $generatedText = $response[0]['generated_text'] ?? '';

        if (empty(trim($generatedText))) {
            return [
                'success' => false,
                'error'   => 'Empty response from model',
                'count'   => 0,
            ];
        }

        Log::debug('AI Response', [
            'length' => strlen($generatedText),
            'preview' => substr($generatedText, 0, 500),
        ]);

        $questions = $this->parseAndInsertQuestions(
            $generatedText,
            $subjectId,
            $subtopicId,
            $uploadedBy,
            $schoolId,
            $count
        );

        return [
            'success'   => true,
            'count'     => count($questions),
            'questions' => $questions,
        ];
    }

    /**
     * Build strict JSON-only prompt matching the new format
     */
    protected function buildPrompt(int $count, string $subtopic, string $subject): string
    {
        return <<<PROMPT
You are a medical education expert. Generate exactly {$count} unique multiple-choice questions about "{$subtopic}" in {$subject}.

CRITICAL REQUIREMENTS:
1. Ensure no questions are repeated from any previous generations
2. Each question must have exactly 4 options
3. Options should NOT include prefixes like "A.", "B.", "C.", or "D." - just the option text
4. Vary difficulty: approximately 1/3 Easy, 1/3 Medium, 1/3 Hard
5. Provide friendly, clear, and concise explanations that:
   - State the correct answer and why it's correct
   - Explain why the other options are incorrect
   - Do NOT reference option letters (A, B, C, D)
   - Do NOT repeat the option text in the explanation

Output ONLY a valid JSON array. No markdown, no code blocks, no preamble.

Format:
[
  {
    "question": "Clinical question here?",
    "options": [
      "First option text without prefix",
      "Second option text without prefix",
      "Third option text without prefix",
      "Fourth option text without prefix"
    ],
    "correct_answer": "First option text without prefix",
    "explanation": "Friendly explanation stating the correct answer is [answer] because [reason]. The other options are incorrect because [reasons for each].",
    "difficulty": "Easy"
  }
]

IMPORTANT:
- Exactly {$count} questions
- Options array has 4 strings WITHOUT letter prefixes
- correct_answer must match one of the options exactly
- Explanation should be conversational and educational
- Keep explanations under 150 words
- Valid JSON only (no trailing commas)
- Start with [ and end with ]

Generate now:
PROMPT;
    }

    /**
     * Parse and insert MCQs with the new format
     */
    protected function parseAndInsertQuestions(
        string $generatedText,
        int $subjectId,
        int $subtopicId,
        int $uploadedBy,
        int $schoolId,
        int $requestedCount = 0
    ): array {
        $parsed = $this->extractQuestionsFromResponse($generatedText);

        if (empty($parsed)) {
            Log::error('AI returned no valid MCQs', [
                'response_length' => strlen($generatedText),
                'snippet' => substr($generatedText, 0, 1000),
            ]);
            return [];
        }

        $recoveredCount = count($parsed);
        
        if ($recoveredCount < $requestedCount && $requestedCount > 0) {
            Log::warning("Response was truncated: recovered only {$recoveredCount}/{$requestedCount} questions", [
                'response_length' => strlen($generatedText),
                'suggestion' => 'Consider using smaller batch sizes or increasing max_tokens',
            ]);
        } else {
            Log::info("Successfully parsed all {$recoveredCount} MCQs");
        }

        DB::beginTransaction();
        $inserted = [];

        try {
            foreach ($parsed as $index => $data) {

                if (!$this->isValidQuestionData($data)) {
                    Log::warning("Invalid MCQ skipped", [
                        'index' => $index,
                        'data' => $data,
                    ]);
                    continue;
                }

                $options = $data['options'];
                $correctAnswer = $data['correct_answer'];

                // Build answer JSON in the new format
                $answerJson = json_encode([
                    'options' => $options,
                    'correct_answer' => $correctAnswer,
                ]);

                $question = Questions::create([
                    'question'      => $data['question'],
                    'answer'        => $answerJson,
                    'explanation'   => $data['explanation'],
                    'difficulty'    => $this->normalizeDifficulty($data['difficulty'] ?? 'Medium'),
                    'question_type' => 'multiple_choice',
                    'subject_id'    => $subjectId,
                    'subtopic_id'   => $subtopicId,
                    'uploaded_by'   => $uploadedBy,
                    'school_id'     => $schoolId,
                    'source'        => 'ai_generated_hf',
                ]);

                $inserted[] = $question;
            }

            DB::commit();
            Log::info('Successfully inserted MCQs', ['count' => count($inserted)]);
            return $inserted;

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Failed inserting MCQs', ['error' => $e->getMessage()]);
            throw $e;
        }
    }

    /**
     * Extract JSON array from AI output
     */
    protected function extractQuestionsFromResponse(string $response): array
    {
        // Remove markdown code blocks if present
        $response = preg_replace('/```json\s*/', '', $response);
        $response = preg_replace('/```\s*/', '', $response);
        
        // Try to extract complete JSON array first
        if (preg_match('/\[\s*\{[\s\S]*\}\s*\]/s', $response, $matches)) {
            $json = trim($matches[0]);
            
            // Clean up common JSON issues
            $json = preg_replace('/,\s*]/', ']', $json);
            $json = preg_replace('/,\s*}/', '}', $json);
            
            $decoded = json_decode($json, true);
            
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                Log::info('Successfully parsed complete JSON array', ['count' => count($decoded)]);
                return $decoded;
            }
        }
        
        // If complete array failed, try to recover complete questions
        Log::warning('Complete JSON parsing failed, attempting to recover complete questions');
        
        return $this->recoverCompleteQuestions($response);
    }

    /**
     * Recover only complete, valid questions from truncated response
     */
    protected function recoverCompleteQuestions(string $response): array
    {
        $questions = [];
        
        // Find all complete question objects
        preg_match_all(
            '/\{[^}]*"question"[^}]*"options"[^}]*"correct_answer"[^}]*"explanation"[^}]*"difficulty"[^}]*\}/s',
            $response,
            $matches
        );
        
        if (empty($matches[0])) {
            Log::error('No complete question objects found in response');
            return [];
        }
        
        foreach ($matches[0] as $index => $questionJson) {
            $decoded = json_decode($questionJson, true);
            
            if (json_last_error() === JSON_ERROR_NONE && $this->isValidQuestionData($decoded)) {
                $questions[] = $decoded;
                Log::debug("Recovered complete question #{$index}");
            } else {
                Log::warning("Question #{$index} incomplete or invalid", [
                    'error' => json_last_error_msg(),
                    'snippet' => substr($questionJson, 0, 200),
                ]);
            }
        }
        
        Log::info('Recovery complete', [
            'found_objects' => count($matches[0]),
            'valid_questions' => count($questions),
        ]);
        
        return $questions;
    }

    /**
     * Validate MCQ structure with new format
     */
    protected function isValidQuestionData(array $data): bool
    {
        // Check if required fields exist
        if (!isset(
            $data['question'],
            $data['options'],
            $data['correct_answer'],
            $data['explanation']
        )) {
            return false;
        }

        // Validate options is an array with exactly 4 elements
        if (!is_array($data['options']) || count($data['options']) !== 4) {
            return false;
        }

        // Ensure all options are strings
        foreach ($data['options'] as $option) {
            if (!is_string($option) || empty(trim($option))) {
                return false;
            }
        }

        // Validate correct_answer matches one of the options
        if (!in_array($data['correct_answer'], $data['options'])) {
            return false;
        }

        // Validate question and explanation are strings
        if (!is_string($data['question']) || !is_string($data['explanation'])) {
            return false;
        }

        return true;
    }

    /**
     * Normalize difficulty values
     */
    protected function normalizeDifficulty(?string $difficulty): string
    {
        return match (strtolower($difficulty ?? 'medium')) {
            'easy', 'simple' => 'Easy',
            'hard', 'difficult' => 'Hard',
            default => 'Medium',
        };
    }

    /**
     * Batch generation with automatic batching for large requests
     */
    public function generateBatch(int $totalCount, array $params): array
    {
        // If requesting more than 5 questions, batch them
        if ($totalCount > 5) {
            $batchSize = 5;
            $batches = ceil($totalCount / $batchSize);
            $allQuestions = [];
            
            for ($i = 0; $i < $batches; $i++) {
                $params['count'] = min($batchSize, $totalCount - ($i * $batchSize));
                $result = $this->generateQuestions($params);
                
                if ($result['success']) {
                    $allQuestions = array_merge($allQuestions, $result['questions']);
                }
                
                // Brief pause between batches to avoid rate limits
                if ($i < $batches - 1) {
                    sleep(2);
                }
            }
            
            return [
                'success' => true,
                'count' => count($allQuestions),
                'questions' => $allQuestions,
            ];
        }
        
        $params['count'] = $totalCount;
        return $this->generateQuestions($params);
    }
}