import React, { useState } from 'react';
import { ChevronDown, ChevronUp, School, Upload, Users, Shield } from 'lucide-react';

function FAQ() {
  const [openItems, setOpenItems] = useState(new Set());
  const [activeCategory, setActiveCategory] = useState('general');

  const toggleItem = (index) => {
    const newOpenItems = new Set(openItems);
    if (newOpenItems.has(index)) {
      newOpenItems.delete(index);
    } else {
      newOpenItems.add(index);
    }
    setOpenItems(newOpenItems);
  };

  const faqData = {
    general: [
      {
        question: "What is included in the free trial?",
        answer: "Your 3-day free trial includes unlimited access to all premium features: thousands of practice questions, comprehensive study guides, AI-powered tutoring, detailed progress tracking, and offline access. No credit card required to start."
      },
      {
        question: "How does the AI tutoring work?",
        answer: "Our AI tutor analyzes your performance patterns and provides personalized explanations for each question. It identifies your weak areas and suggests targeted study materials, adapts to your learning style, and offers 24/7 support whenever you need help understanding complex medical concepts."
      },
      {
        question: "Can I cancel anytime?",
        answer: "Absolutely! You can cancel your subscription at any time during or after your free trial with no penalties or hidden fees. If you cancel during the trial period, you won't be charged anything."
      },
      {
        question: "Is my data secure and private?",
        answer: "Yes, we use enterprise-grade security measures including end-to-end encryption, secure cloud storage, and strict privacy policies. Your study data and personal information are never shared with third parties without your explicit consent."
      },
     
    ],
    collaboration: [
      {
        question: "How can medical schools partner with us?",
        answer: "Medical schools can collaborate with us through our Academic Partnership Program. We offer custom integration solutions, bulk licensing for students, branded platforms, and the ability to upload your institution's proprietary question banks while maintaining full control over your content."
      },
      {
        question: "Can schools upload their own MCQ questions?",
        answer: "Yes! Through our secure institutional portal, medical schools can upload their proprietary MCQ questions in various formats (PDF, Word, Excel, or through our web interface). All uploaded content goes through our quality review process and remains exclusive to your institution's students unless you choose to share more broadly."
      },
      {
        question: "What formats do you accept for question uploads?",
        answer: "We accept multiple formats including: PDF documents, Microsoft Word files, Excel spreadsheets, CSV files, and direct entry through our web portal. Our system can parse most standard question formats and automatically categorize them by subject, difficulty, and medical specialty."
      },
      {
        question: "How is uploaded content reviewed and quality-controlled?",
        answer: "All uploaded questions go through a rigorous 3-step review process: automated content scanning for formatting and completeness, expert medical educator review for accuracy and clarity, and final quality assurance testing. Schools retain full editorial control and can request revisions at any stage."
      },
     
      {
        question: "What are the costs for institutional partnerships?",
        answer: "Partnership costs vary based on your needs: student population size, content volume, customization requirements, and support level. We offer flexible pricing models including per-student licensing, flat-rate institutional access, and revenue-sharing for broadly shared content. Contact us for a custom quote."
      },
      {
        question: "How do we maintain control over our proprietary content?",
        answer: "You retain complete ownership and control of your content. We provide granular permissions allowing you to: restrict access to your students only, set time-limited availability, control sharing permissions, request content removal at any time, and maintain version control over updates."
      },
      
    ]
  };

  const categories = [
    { id: 'general', label: 'General Questions', icon: Users, count: faqData.general.length },
    { id: 'collaboration', label: 'Medical School Partnership', icon: School, count: faqData.collaboration.length }
  ];

  return (
    <section className="bg-white py-24 relative overflow-hidden">
      {/* Background Effects */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-10 right-10 w-40 h-40 bg-gradient-to-r from-blue-500/5 to-purple-500/5 rounded-full blur-3xl"></div>
        <div className="absolute bottom-20 left-10 w-32 h-32 bg-gradient-to-r from-indigo-500/5 to-cyan-500/5 rounded-full blur-2xl"></div>
        <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-60 h-60 bg-gradient-to-r from-purple-500/3 to-pink-500/3 rounded-full blur-3xl"></div>
      </div>

      <div className="max-w-6xl mx-auto px-4 relative z-10">
        {/* Header */}
        <div className="text-center mb-16">
          <div className="inline-block mb-4">
            <span className="bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent text-sm font-semibold tracking-wide uppercase">
              Frequently Asked Questions
            </span>
          </div>
          <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6 tracking-tight">
            Got Questions?
            <span className="block bg-gradient-to-r from-blue-700 to-cyan-700 bg-clip-text text-transparent">
              We've Got Answers
            </span>
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
            Find answers to common questions about our platform, features, and how medical schools can collaborate with us to enhance their students' learning experience.
          </p>
        </div>

        {/* Category Tabs */}
        <div className="flex flex-wrap justify-center gap-4 mb-12">
          {categories.map((category) => {
            const Icon = category.icon;
            return (
              <button
                key={category.id}
                onClick={() => setActiveCategory(category.id)}
                className={`flex items-center px-6 py-3 rounded-2xl font-semibold transition-all duration-300 transform hover:scale-105 ${
                  activeCategory === category.id
                    ? 'bg-gradient-to-r from-blue-700 to-cyan-700 text-white shadow-lg'
                    : 'bg-white text-gray-700 hover:bg-gray-50 border border-gray-200 shadow-sm'
                }`}
              >
                <Icon className="w-5 h-5 mr-2" />
                {category.label}
                <span className={`ml-2 px-2 py-1 text-xs rounded-full ${
                  activeCategory === category.id 
                    ? 'bg-white/20 text-white' 
                    : 'bg-gray-100 text-gray-600'
                }`}>
                  {category.count}
                </span>
              </button>
            );
          })}
        </div>

        

        {/* FAQ Items */}
        <div className="max-w-4xl mx-auto">
          <div className="space-y-4">
            {faqData[activeCategory].map((item, index) => {
              const isOpen = openItems.has(index);
              return (
                <div
                  key={index}
                  className="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden hover:shadow-md transition-all duration-300"
                >
                  <button
                    onClick={() => toggleItem(index)}
                    className="w-full px-8 py-6 text-left flex items-center justify-between hover:bg-gray-50 transition-colors duration-200"
                  >
                    <h3 className="text-lg font-semibold text-gray-900 pr-4">
                      {item.question}
                    </h3>
                    <div className={`flex-shrink-0 transition-transform duration-300 ${
                      isOpen ? 'rotate-180' : ''
                    }`}>
                      <ChevronDown className="w-5 h-5 text-gray-500" />
                    </div>
                  </button>
                  
                  <div className={`overflow-hidden transition-all duration-300 ease-out ${
                    isOpen ? 'max-h-96 opacity-100' : 'max-h-0 opacity-0'
                  }`}>
                    <div className="px-8 pb-6">
                      <div className="h-px bg-gradient-to-r from-transparent via-gray-200 to-transparent mb-4"></div>
                      <p className="text-gray-700 leading-relaxed">
                        {item.answer}
                      </p>
                    </div>
                  </div>
                </div>
              );
            })}
          </div>
        </div>
      </div>

      {/* Floating Elements */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        {[...Array(8)].map((_, i) => (
          <div
            key={i}
            className="absolute w-1 h-1 bg-gradient-to-r from-blue-400 to-purple-400 rounded-full animate-pulse"
            style={{
              left: `${Math.random() * 100}%`,
              top: `${Math.random() * 100}%`,
              animationDelay: `${Math.random() * 3}s`,
              animationDuration: `${3 + Math.random() * 2}s`
            }}
          ></div>
        ))}
      </div>
    </section>
  );
}

export default FAQ;