import React, { useState, useEffect, useRef } from 'react';

function Features() {
  const [isVisible, setIsVisible] = useState(false);
  const [hoveredCard, setHoveredCard] = useState(null);
  const sectionRef = useRef(null);

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true);
        }
      },
      { threshold: 0.1 }
    );

    if (sectionRef.current) {
      observer.observe(sectionRef.current);
    }

    return () => observer.disconnect();
  }, []);

  const features = [
    {
      title: "Practice Tests",
      description: "Access thousands of exam-style questions tailored for medical students with instant feedback.",
      icon: "📝",
      gradient: "from-blue-500 to-cyan-500",
      bgGradient: "from-blue-50 to-cyan-50",
      details: ["10,000+ Questions", "Instant Feedback", "Timed Exams", "Performance Analytics"]
    },
    {
      title: "Study Guides",
      description: "Curated resources and visual aids to simplify complex medical topics and concepts.",
      icon: "📚",
      gradient: "from-purple-500 to-pink-500",
      bgGradient: "from-purple-50 to-pink-50",
      details: ["Visual Learning", "Step-by-Step Guides", "Interactive Content", "Expert Curated"]
    },
    {
      title: "Progress Tracking",
      description: "Monitor your study progress with detailed analytics and personalized recommendations.",
      icon: "📊",
      gradient: "from-green-500 to-emerald-500",
      bgGradient: "from-green-50 to-emerald-50",
      details: ["Real-time Analytics", "Goal Setting", "Weakness Analysis", "Study Streaks"]
    },
    {
      title: "AI-Powered Learning",
      description: "Personalized study paths and smart recommendations powered by artificial intelligence.",
      icon: "🤖",
      gradient: "from-orange-500 to-red-500",
      bgGradient: "from-orange-50 to-red-50",
      details: ["Smart Recommendations", "Adaptive Learning", "Personalized Paths", "AI Tutoring"]
    },
    // {
    //   title: "Collaborative Study",
    //   description: "Connect with peers, join study groups, and share knowledge with the community.",
    //   icon: "👥",
    //   gradient: "from-indigo-500 to-purple-500",
    //   bgGradient: "from-indigo-50 to-purple-50",
    //   details: ["Study Groups", "Peer Discussion", "Knowledge Sharing", "Mentor Connect"]
    // },
    // {
    //   title: "Mobile Learning",
    //   description: "Study anywhere, anytime with our responsive mobile app and offline capabilities.",
    //   icon: "📱",
    //   gradient: "from-teal-500 to-blue-500",
    //   bgGradient: "from-teal-50 to-blue-50",
    //   details: ["Mobile App", "Offline Mode", "Sync Across Devices", "Touch Optimized"]
    // }
  ];

  return (
    <section ref={sectionRef} className="relative bg-white py-20 lg:py-32 overflow-hidden">
      
      {/* Background Decorative Elements */}
      <div className="absolute inset-0 overflow-hidden">
        <div className="absolute -top-40 -right-40 w-80 h-80 bg-gradient-to-br from-blue-200 to-purple-200 rounded-full opacity-20 animate-pulse" />
        <div className="absolute -bottom-40 -left-40 w-96 h-96 bg-gradient-to-tr from-cyan-200 to-blue-200 rounded-full opacity-20 animate-pulse" style={{ animationDelay: '1s' }} />
        
        {/* Floating Shapes */}
        {[...Array(8)].map((_, i) => (
          <div
            key={i}
            className="absolute opacity-10 animate-float"
            style={{
              left: `${10 + (i * 12)}%`,
              top: `${20 + (i * 8)}%`,
              animationDelay: `${i * 0.5}s`,
              animationDuration: `${4 + (i % 3)}s`
            }}
          >
            <div className={`w-4 h-4 ${i % 2 === 0 ? 'bg-blue-400' : 'bg-purple-400'} ${i % 3 === 0 ? 'rounded-full' : 'rotate-45'}`} />
          </div>
        ))}
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
        
        {/* Header Section */}
        <div className={`text-center mb-16 lg:mb-20 transform transition-all duration-1000 ${
          isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'
        }`}>
          <div className="inline-flex items-center px-4 py-2 bg-white/80 backdrop-blur-sm rounded-full border border-blue-200 shadow-sm mb-6">
            <span className="w-2 h-2 bg-blue-500 rounded-full mr-2 animate-pulse" />
            <span className="text-sm font-medium text-blue-700">Powerful Features</span>
          </div>
          
          <h2 className="text-4xl sm:text-5xl lg:text-6xl font-bold text-gray-900 mb-6">
            <span className="block">Why Choose</span>
            <span className="bg-gradient-to-r from-blue-700 to-cyan-700 bg-clip-text text-transparent">
              MedArchieve?
            </span>
          </h2>
          
          <p className="text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed">
            Discover the comprehensive suite of tools designed to accelerate your medical education journey
          </p>
        </div>

        {/* Features Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 lg:gap-8">
          {features.map((feature, index) => (
            <div
              key={index}
              className={`group relative transform transition-all duration-700 ${
                isVisible ? 'translate-y-0 opacity-100' : 'translate-y-20 opacity-0'
              }`}
              style={{ transitionDelay: `${index * 0.1}s` }}
              onMouseEnter={() => setHoveredCard(index)}
              onMouseLeave={() => setHoveredCard(null)}
            >
              {/* Card Background with Gradient */}
              <div className={`absolute inset-0 bg-gradient-to-br ${feature.bgGradient} rounded-2xl transform transition-all duration-500 ${
                hoveredCard === index ? 'scale-105 rotate-1' : 'scale-100'
              }`} />
              
              {/* Card Content */}
              <div className="relative bg-white/80 backdrop-blur-sm rounded-2xl p-6 lg:p-8 border border-white/50 shadow-xl hover:shadow-2xl transition-all duration-500 h-full flex flex-col">
                
                {/* Icon and Header */}
                <div className="flex items-start justify-between mb-6">
                  <div className={`text-4xl lg:text-5xl transform transition-all duration-500 ${
                    hoveredCard === index ? 'scale-110 rotate-12' : 'scale-100'
                  }`}>
                    {feature.icon}
                  </div>
                  
                  {/* Decorative Gradient Dot */}
                  <div className={`w-3 h-3 bg-gradient-to-r ${feature.gradient} rounded-full animate-pulse`} />
                </div>

                {/* Title */}
                <h3 className={`text-2xl font-bold text-gray-900 mb-4 transform transition-all duration-300 ${
                  hoveredCard === index ? 'text-transparent bg-gradient-to-r bg-clip-text ' + feature.gradient : ''
                }`}>
                  {feature.title}
                </h3>

                {/* Description */}
                <p className="text-gray-600 leading-relaxed mb-6">
                  {feature.description}
                </p>

                {/* Feature Details - Always visible */}
                <div className="space-y-2 mb-6">
                  {feature.details.map((detail, detailIndex) => (
                    <div key={detailIndex} className="flex items-center text-sm text-gray-500">
                      <div className={`w-1.5 h-1.5 bg-gradient-to-r ${feature.gradient} rounded-full mr-3 animate-pulse`} 
                           style={{ animationDelay: `${detailIndex * 0.1}s` }} />
                      {detail}
                    </div>
                  ))}
                </div>

                {/* Action Button - Always visible */}
                <div className="mt-auto">
                  <button className={`w-full py-3 px-4 bg-gradient-to-r ${feature.gradient} text-white font-semibold rounded-xl transform transition-all duration-300 hover:scale-105 hover:shadow-lg`}>
                    Learn More
                  </button>
                </div>

                {/* Floating Animation Elements */}
                {hoveredCard === index && (
                  <>
                    <div className="absolute -top-2 -right-2 w-4 h-4 bg-yellow-400 rounded-full animate-ping" />
                    <div className="absolute -bottom-2 -left-2 w-3 h-3 bg-green-400 rounded-full animate-bounce" />
                  </>
                )}
              </div>
            </div>
          ))}
        </div>

        {/* Bottom CTA Section */}
        <div className={`text-center mt-16 lg:mt-20 transform transition-all duration-1000 ${
          isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'
        }`} style={{ transitionDelay: '0.8s' }}>
          <div className="bg-white/80 backdrop-blur-sm rounded-2xl p-8 border border-white/50 shadow-xl max-w-2xl mx-auto">
            <h3 className="text-2xl font-bold text-gray-900 mb-4">
              Ready to Transform Your Medical Studies?
            </h3>
            <p className="text-gray-600 mb-6">
              Join thousands of medical students who are already excelling with MedArchieve
            </p>
            <button className="group inline-flex items-center px-8 py-4 bg-gradient-to-r from-blue-700 to-cyan-700 text-white font-semibold rounded-full transform transition-all duration-300 hover:scale-105 hover:shadow-2xl">
              Get Started Today
              <svg className="ml-2 w-5 h-5 transform group-hover:translate-x-1 transition-transform duration-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7l5 5m0 0l-5 5m5-5H6" />
              </svg>
            </button>
          </div>
        </div>
      </div>

      {/* Custom Animations */}
      <style jsx>{`
        @keyframes float {
          0%, 100% {
            transform: translateY(0px) rotate(0deg);
          }
          33% {
            transform: translateY(-10px) rotate(120deg);
          }
          66% {
            transform: translateY(5px) rotate(240deg);
          }
        }
        
        @keyframes fadeIn {
          from {
            opacity: 0;
            transform: translateY(10px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
        
        .animate-float {
          animation: float ease-in-out infinite;
        }
        
        .animate-fadeIn {
          animation: fadeIn 0.3s ease-out forwards;
        }
      `}</style>
    </section>
  );
}

export default Features;