import React, { useState, useEffect } from 'react';
import { FcReading, FcSalesPerformance, FcRating } from 'react-icons/fc';

import image1 from '../assets/img2.jpg';
import dots from '../assets/bg_dots.svg';
import shapes from '../assets/shapes.svg';

function Hero() {
  const [isVisible, setIsVisible] = useState(false);
  const [mousePosition, setMousePosition] = useState({ x: 0, y: 0 });

  useEffect(() => {
    setIsVisible(true);
    
    const handleMouseMove = (e) => {
      setMousePosition({
        x: (e.clientX / window.innerWidth) * 100,
        y: (e.clientY / window.innerHeight) * 100,
      });
    };

    window.addEventListener('mousemove', handleMouseMove);
    return () => window.removeEventListener('mousemove', handleMouseMove);
  }, []);

  const stats = [
    { icon: FcReading, number: '500+', label: 'Registered Users', delay: '0s' },
    { icon: FcSalesPerformance, number: '95%', label: 'Success Rate', delay: '0.3s' },
    { icon: FcRating, number: '4.9', label: 'User Rating', delay: '0.6s' }
  ];

  return (
    <section className="relative pt-24 bg-gradient-to-br from-blue-50 via-cyan-50 to-blue-100 min-h-screen flex items-center overflow-hidden">
      {/* Animated Background Elements */}
      <div 
        className="absolute inset-0 opacity-30"
        style={{
          background: `radial-gradient(circle at ${mousePosition.x}% ${mousePosition.y}%, rgba(56, 189, 248, 0.1) 0%, transparent 50%)`
        }}
      />
      
      {/* Floating Particles */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        {[...Array(15)].map((_, i) => (
          <div
            key={i}
            className="absolute animate-pulse opacity-20"
            style={{
              left: `${Math.random() * 100}%`,
              top: `${Math.random() * 100}%`,
              animationDelay: `${Math.random() * 2}s`,
              animationDuration: `${3 + Math.random() * 2}s`
            }}
          >
            <div className="w-2 h-2 bg-blue-400 rounded-full animate-bounce" />
          </div>
        ))}
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-8 lg:gap-16 items-center">
          
          {/* Text Content */}
          <div className={`space-y-6 lg:space-y-8 transform transition-all duration-1000 ${
            isVisible ? 'translate-x-0 opacity-100' : '-translate-x-20 opacity-0'
          }`}>
            
            {/* Badge */}
            <div className="inline-flex items-center px-4 py-2 bg-white/80 backdrop-blur-sm rounded-full border border-blue-200 shadow-sm animate-fadeInDown">
              <span className="w-2 h-2 bg-green-400 rounded-full mr-2 animate-pulse" />
              <span className="text-sm font-medium text-blue-700">Join 500+ Medical Students</span>
            </div>

            {/* Main Heading */}
            <h1 className="text-4xl sm:text-5xl lg:text-6xl xl:text-7xl font-bold text-gray-900 leading-tight">
              <span className="block animate-fadeInUp" style={{ animationDelay: '0.2s' }}>
                Master Your
              </span>
              <span className="block bg-gradient-to-r from-blue-600 to-cyan-600 bg-clip-text text-transparent animate-fadeInUp" style={{ animationDelay: '0.4s' }}>
                Medical Exams
              </span>
              <span className="block text-2xl sm:text-3xl lg:text-4xl font-medium text-gray-600 mt-2 animate-fadeInUp" style={{ animationDelay: '0.6s' }}>
                with MedArchieve
              </span>
            </h1>

            {/* Description */}
            <p className={`text-lg sm:text-xl text-gray-600 max-w-lg leading-relaxed transform transition-all duration-700 ${
              isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'
            }`} style={{ transitionDelay: '0.8s' }}>
              Empowering medical students with expertly curated resources, interactive practice tests, and AI-powered study tools to excel in their exams.
            </p>

            {/* CTA Buttons */}
            <div className={`flex flex-col sm:flex-row gap-4 transform transition-all duration-700 ${
              isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'
            }`} style={{ transitionDelay: '1s' }}>
              <button className="group relative px-8 py-4 bg-gradient-to-r from-blue-600 to-cyan-600 text-white font-semibold rounded-full overflow-hidden transition-all duration-300 hover:scale-105 hover:shadow-2xl transform hover:-translate-y-1">
                <div className="absolute inset-0 bg-gradient-to-r from-blue-700 to-cyan-700 transform scale-x-0 group-hover:scale-x-100 transition-transform duration-300 origin-left" />
                <span className="relative flex items-center justify-center">
                  Start Learning
                  <svg className="ml-2 w-5 h-5 transform group-hover:translate-x-1 transition-transform duration-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7l5 5m0 0l-5 5m5-5H6" />
                  </svg>
                </span>
              </button>
              
              <button className="group px-8 py-4 border-2 border-blue-600 text-blue-600 font-semibold rounded-full transition-all duration-300 hover:bg-blue-600 hover:text-white hover:scale-105 hover:shadow-lg transform hover:-translate-y-1">
                <span className="flex items-center justify-center">
                  Watch Demo
                  <svg className="ml-2 w-5 h-5 transform group-hover:scale-110 transition-transform duration-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14.828 14.828a4 4 0 01-5.656 0M9 10h1m4 0h1m-6 4h.01M19 10a9 9 0 11-18 0 9 9 0 0118 0z" />
                  </svg>
                </span>
              </button>
            </div>

            {/* Trust Indicators */}
            <div className={`flex flex-wrap items-center gap-6 pt-4 transform transition-all duration-700 ${
              isVisible ? 'translate-y-0 opacity-100' : 'translate-y-10 opacity-0'
            }`} style={{ transitionDelay: '1.2s' }}>
              <div className="flex items-center space-x-1">
                <div className="flex">
                  {[...Array(5)].map((_, i) => (
                    <svg key={i} className="w-5 h-5 text-yellow-400 fill-current" viewBox="0 0 20 20">
                      <path d="M10 15l-5.878 3.09 1.123-6.545L.489 6.91l6.572-.955L10 0l2.939 5.955 6.572.955-4.756 4.635 1.123 6.545z"/>
                    </svg>
                  ))}
                </div>
                <span className="text-sm text-gray-600 ml-2">4.9/5 from 200+ reviews</span>
              </div>
            </div>
          </div>

          {/* Image Section */}
          <div className={`relative transform transition-all duration-1000 ${
            isVisible ? 'translate-x-0 opacity-100' : 'translate-x-20 opacity-0'
          }`} style={{ transitionDelay: '0.5s' }}>
            
            {/* Main Image Container */}
            <div className="relative max-w-lg mx-auto lg:max-w-none">
              
              {/* Background Decorative Elements */}
              <div className="absolute inset-0 bg-gradient-to-r from-blue-400 to-cyan-400 rounded-3xl transform rotate-6 scale-105 opacity-20 animate-pulse" />
              <div className="absolute inset-0 bg-gradient-to-l from-purple-400 to-pink-400 rounded-3xl transform -rotate-6 scale-95 opacity-10 animate-pulse" style={{ animationDelay: '1s' }} />
              
              {/* Dots Background */}
              <img
                src={dots}
                alt=""
                className="absolute -top-4 -left-4 w-full h-full object-cover rounded-2xl opacity-30 animate-pulse"
              />
              
              {/* Main Image */}
              <div className="relative bg-white rounded-2xl p-4 shadow-2xl transform hover:scale-105 transition-all duration-500 hover:rotate-1">
                <img
                  src={image1}
                  alt="Medical student studying with digital tools"
                  className="w-full h-[30rem] object-cover rounded-xl"
                />
                
                {/* Floating UI Elements */}
                <div className="absolute -top-2 -right-2 bg-green-500 text-white p-2 rounded-full animate-bounce">
                  <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                    <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                  </svg>
                </div>
              </div>

              {/* Decorative Shapes */}
              <img
                src={shapes}
                alt=""
                className="absolute -bottom-4 -right-4 w-16 h-16 opacity-60 animate-spin"
                style={{ animationDuration: '10s' }}
              />
            </div>

            {/* Floating Stats Cards */}
            <div className="absolute inset-0 pointer-events-none">
              {stats.map((stat, index) => {
                const positions = [
                  { top: '10%', left: '-8%', sm: { top: '5%', left: '-12%' } },
                  { top: '40%', right: '-8%', sm: { top: '35%', right: '-12%' } },
                  { bottom: '1%', left: '5%', sm: { bottom: '10%', left: '0%' } }
                ];
                
                return (
                  <div
                    key={index}
                    className={`absolute bg-white rounded-xl shadow-xl border border-gray-100 p-2 sm:p-4 transform transition-all duration-700 hover:scale-110 hover:shadow-2xl animate-float ${
                      isVisible ? 'translate-y-0 opacity-100' : 'translate-y-20 opacity-0'
                    }`}
                    style={{
                      ...positions[index],
                      animationDelay: stat.delay,
                      transitionDelay: `${1.5 + index * 0.2}s`
                    }}
                  >
                    <div className="text-center">
                      <div className="inline-flex items-center justify-center w-12 h-12 bg-gradient-to-br from-blue-50 to-cyan-50 rounded-xl mb-3">
                        <stat.icon className="text-2xl" />
                      </div>
                      <div className="text-xl sm:text-2xl font-bold text-gray-900 mb-1">
                        {stat.number}
                      </div>
                      <div className="text-xs sm:text-sm text-gray-600 font-medium">
                        {stat.label}
                      </div>
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        </div>
      </div>

      {/* Custom Animations */}
      <style jsx>{`
        @keyframes fadeInUp {
          from {
            opacity: 0;
            transform: translateY(30px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
        
        @keyframes fadeInDown {
          from {
            opacity: 0;
            transform: translateY(-30px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
        
        @keyframes float {
          0%, 100% {
            transform: translateY(0px);
          }
          50% {
            transform: translateY(-10px);
          }
        }
        
        .animate-fadeInUp {
          animation: fadeInUp 0.8s ease-out forwards;
          opacity: 0;
        }
        
        .animate-fadeInDown {
          animation: fadeInDown 0.8s ease-out forwards;
          opacity: 0;
        }
        
        .animate-float {
          animation: float 3s ease-in-out infinite;
        }
      `}</style>
    </section>
  );
}

export default Hero;