import React, { useState, useEffect } from 'react';
import { useStats } from './Dashboard';
import { 
  BookOpen, 
  CheckCircle, 
  XCircle, 
  Brain, 
  Target, 
  ArrowLeft,
  Clock,
  User,
  Building,
  ThumbsUp,
  ThumbsDown,
  Eye,
  Calendar
} from 'lucide-react';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL;

const AdminDashboard = () => {
  const { userProfile } = useStats();
  const [pendingQuestions, setPendingQuestions] = useState([]);
  const [stats, setStats] = useState({ pending: 0, approved: 0, rejected: 0 });
  const [subjects, setSubjects] = useState([]);
  const [selectedQuestion, setSelectedQuestion] = useState(null);
  const [editedQuestion, setEditedQuestion] = useState(null);
  const [reviewMode, setReviewMode] = useState('list');
  const [searchTerm, setSearchTerm] = useState('');

  useEffect(() => {
    const fetchData = async () => {
      try {
        const token = localStorage.getItem('auth_token');
        console.log('Auth Token:', token);
        console.log('User Profile:', userProfile);

        if (!token) {
          console.error('No auth token found');
          return;
        }
        if (!userProfile.institution) {
          console.error('No school_id found in userProfile');
          return;
        }

        const headers = {
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json',
          Accept: 'application/json',
        };

        console.log('Fetching from:', `${API_BASE_URL}/admin/questions/pending?school_id=${userProfile.institution}`);
        const [pendingRes, statsRes, subjectsRes] = await Promise.all([
          fetch(`${API_BASE_URL}/admin/questions/pending?school_id=${userProfile.institution}`, { headers }),
          fetch(`${API_BASE_URL}/admin/questions/count?school_id=${userProfile.institution}`, { headers }),
          fetch(`${API_BASE_URL}/dashboard/subjects`, { headers }),
        ]);

        console.log('Pending Response Status:', pendingRes.status, pendingRes.statusText);
        console.log('Stats Response Status:', statsRes.status, statsRes.statusText);
        console.log('Subjects Response Status:', subjectsRes.status, subjectsRes.statusText);

        if (!pendingRes.ok || !statsRes.ok || !subjectsRes.ok) {
          throw new Error('One or more API requests failed');
        }

        const [pendingData, statsData, subjectsData] = await Promise.all([
          pendingRes.json(),
          statsRes.json(),
          subjectsRes.json(),
        ]);

        console.log('Pending Data:', pendingData);
        console.log('Stats Data:', statsData);
        console.log('Subjects Data:', subjectsData);

        setPendingQuestions(pendingData.questions || []);
        setStats({
          pending: statsData.pending || 0,
          approved: statsData.approved || 0,
          rejected: statsData.rejected || 0,
        });
        setSubjects(subjectsData.data || []);
      } catch (err) {
        console.error('Error fetching data:', err);
      }
    };

    if (userProfile.role === 'admin' && userProfile.institution) {
      fetchData();
    }
  }, [userProfile.role, userProfile.institution]);

  const handleSaveEdits = async (questionId) => {
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        console.error('No auth token found');
        return;
      }

      const payload = {
        ...editedQuestion,
        subject: editedQuestion.subject,
        subtopic: editedQuestion.subtopic || null,
        answer: editedQuestion.question_type === 'multiple_choice' ? JSON.stringify(editedQuestion.answer) : editedQuestion.answer,
        metadata: editedQuestion.metadata ? JSON.stringify(editedQuestion.metadata) : null,
        
      };

      const res = await fetch(`${API_BASE_URL}/admin/questions/${questionId}/update`, {
        method: 'post',
        headers: {
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json',
          Accept: 'application/json',
        },
        body: JSON.stringify(payload),
      });

      if (!res.ok) {
        throw new Error(`HTTP error! Status: ${res.status}`);
      }

      setPendingQuestions(prev =>
        prev.map(q => (q.id === questionId ? { 
          ...q, 
          ...editedQuestion, 
          answer: editedQuestion.question_type === 'multiple_choice' ? JSON.stringify(editedQuestion.answer) : editedQuestion.answer,
          subject: { name: editedQuestion.subject },
          user: { name: q.user.name }
        } : q))
      );
      setSelectedQuestion(editedQuestion);
    } catch (err) {
      console.error('Error saving question edits:', err);
    }
  };

  const handleApproval = async (questionId, status) => {
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        console.error('No auth token found');
        return;
      }

      // If approving and edits exist, save edits first
      if (status === 'approved' && editedQuestion) {
        await handleSaveEdits(questionId);
      }

      const res = await fetch(`${API_BASE_URL}/admin/questions/${questionId}/approve`, {
        method: 'POST',
        headers: {
          Authorization: `Bearer ${token}`,
          'Content-Type': 'application/json',
          Accept: 'application/json',
        },
        body: JSON.stringify({ status }),
      });

      if (!res.ok) {
        throw new Error(`HTTP error! Status: ${res.status}`);
      }

      setPendingQuestions(prev => prev.filter(q => q.id !== questionId));
      setStats(prev => ({
        ...prev,
        pending: prev.pending - 1,
        [status]: prev[status] + 1,
      }));
      setSelectedQuestion(null);
      setEditedQuestion(null);
      setReviewMode('list');
    } catch (err) {
      console.error('Error updating question:', err);
    }
  };

  const getDifficultyColor = (difficulty) => {
    switch (difficulty?.toLowerCase()) {
      case 'easy': return 'from-green-500 to-emerald-500';
      case 'medium': return 'from-yellow-500 to-amber-500';
      case 'hard': return 'from-red-500 to-rose-500';
      default: return 'from-gray-500 to-slate-500';
    }
  };

  const getTypeIcon = (type) => {
    return type === 'multiple_choice' ? Target : BookOpen;
  };

  const formatDate = (dateString) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  if (reviewMode === 'detail' && selectedQuestion) {
    const TypeIcon = getTypeIcon(selectedQuestion.question_type);

    return (
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 p-6">
        <div className="max-w-4xl mx-auto">
          {/* Header */}
          <div className="flex items-center justify-between mb-8">
            <button
              onClick={() => {
                setReviewMode('list');
                setEditedQuestion(null);
              }}
              className="flex items-center space-x-2 text-gray-600 hover:text-gray-800 transition-colors duration-200"
            >
              <ArrowLeft className="h-5 w-5" />
              <span className="font-medium">Back to Questions</span>
            </button>
            <div className="flex items-center space-x-3">
              <div className={`px-4 py-2 bg-gradient-to-r ${getDifficultyColor(editedQuestion?.difficulty || selectedQuestion.difficulty)} text-white rounded-full text-sm font-semibold shadow-lg`}>
                {editedQuestion?.difficulty || selectedQuestion.difficulty}
              </div>
              <div className="px-4 py-2 bg-gradient-to-r from-blue-500 to-cyan-500 text-white rounded-full text-sm font-semibold shadow-lg flex items-center space-x-2">
                <TypeIcon className="h-4 w-4" />
                <span>{(editedQuestion?.question_type || selectedQuestion.question_type).toUpperCase()}</span>
              </div>
            </div>
          </div>

          {/* Question Details Card */}
          <div className="bg-white/95 backdrop-blur-sm rounded-3xl shadow-2xl border border-gray-200 overflow-hidden">
            {/* Question Header */}
            <div className="bg-gradient-to-r from-blue-600 via-purple-600 to-indigo-600 p-8 text-white">
              <div className="flex items-start justify-between mb-6">
                <div className="flex-1">
                  <div className="flex items-center space-x-3 mb-4">
                    <div className="p-3 bg-white/20 backdrop-blur-sm rounded-xl">
                      <Brain className="h-6 w-6" />
                    </div>
                    <div>
                      <h1 className="text-2xl font-bold">Question Review</h1>
                      <p className="text-blue-100">ID: #{selectedQuestion.id}</p>
                    </div>
                  </div>
                </div>
                <div className="text-right">
                  <div className="flex items-center space-x-2 text-blue-100 mb-2">
                    <Calendar className="h-4 w-4" />
                    <span className="text-sm">{formatDate(selectedQuestion.created_at)}</span>
                  </div>
                  <div className="flex items-center space-x-2 text-blue-100">
                    <User className="h-4 w-4" />
                    <span className="text-sm">{selectedQuestion.user.name}</span>
                  </div>
                </div>
              </div>

              
            </div>

            {/* Question Content */}
            <div className="p-8 space-y-8">
              {/* Question Text */}
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <BookOpen className="inline h-5 w-5 mr-2 text-green-600" />
                  Question
                </label>
                <textarea
                  value={editedQuestion?.question || selectedQuestion.question}
                  onChange={(e) => setEditedQuestion({ ...editedQuestion, question: e.target.value })}
                  className="w-full p-6 bg-gradient-to-br from-gray-50 to-gray-100 rounded-2xl border-2 border-gray-200 text-gray-800 text-lg leading-relaxed"
                  rows={4}
                />
              </div>

              {/* MCQ Options */}
              {selectedQuestion.question_type === 'multiple_choice' && (
                <div className="space-y-4">
                  <label className="block text-lg font-semibold text-gray-800">
                    <Target className="inline h-5 w-5 mr-2 text-blue-600" />
                    Answer Options
                  </label>
                  <div className="space-y-3">
                    {(() => {
                      const currentAnswer = editedQuestion?.answer || JSON.parse(selectedQuestion.answer);
                      return currentAnswer.options.map((option, index) => {
                        const isCorrect = option === currentAnswer.correct_answer;
                        return (
                          <div
                            key={index}
                            className={`flex items-center space-x-4 p-4 rounded-2xl border-2 transition-all duration-300 ${
                              isCorrect ? 'border-green-500 bg-green-50 shadow-lg' : 'border-gray-200 bg-white hover:border-gray-300'
                            }`}
                          >
                            <div className={`flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center text-white font-semibold ${
                              isCorrect ? 'bg-gradient-to-br from-green-500 to-emerald-600' : 'bg-gradient-to-br from-blue-500 to-purple-600'
                            }`}>
                              {String.fromCharCode(65 + index)}
                            </div>
                            <input
                              type="text"
                              value={option}
                              onChange={(e) => {
                                const newOptions = [...currentAnswer.options];
                                newOptions[index] = e.target.value;
                                // If this option was the correct answer, update correct_answer
                                const newCorrectAnswer = option === currentAnswer.correct_answer ? e.target.value : currentAnswer.correct_answer;
                                setEditedQuestion({ 
                                  ...editedQuestion, 
                                  answer: { ...currentAnswer, options: newOptions, correct_answer: newCorrectAnswer },
                                  question_type: selectedQuestion.question_type
                                });
                              }}
                              className="flex-1 p-2 border border-gray-300 rounded-lg bg-white text-gray-800"
                            />
                            <button
                              onClick={() => {
                                setEditedQuestion({
                                  ...editedQuestion,
                                  answer: { ...currentAnswer, correct_answer: option },
                                  question_type: selectedQuestion.question_type
                                });
                              }}
                              className={`px-3 py-1 rounded-lg ${isCorrect ? 'bg-green-600 text-white' : 'bg-gray-200 text-gray-800'}`}
                            >
                              {isCorrect ? 'Correct' : 'Set as Correct'}
                            </button>
                          </div>
                        );
                      });
                    })()}
                  </div>
                </div>
              )}

              {/* Viva Answer */}
              {selectedQuestion.question_type === 'viva' && (
                <div className="space-y-4">
                  <label className="block text-lg font-semibold text-gray-800">
                    <BookOpen className="inline h-5 w-5 mr-2 text-purple-600" />
                    Expected Answer
                  </label>
                  <textarea
                    value={editedQuestion?.answer || selectedQuestion.answer}
                    onChange={(e) => setEditedQuestion({ ...editedQuestion, answer: e.target.value, question_type: selectedQuestion.question_type })}
                    className="w-full p-6 bg-gradient-to-br from-purple-50 to-pink-50 rounded-2xl border-2 border-purple-200 text-gray-800 text-lg leading-relaxed"
                    rows={4}
                  />
                </div>
              )}

              {/* Explanation */}
              <div className="space-y-4">
                <label className="block text-lg font-semibold text-gray-800">
                  <Brain className="inline h-5 w-5 mr-2 text-indigo-600" />
                  Explanation
                </label>
                <textarea
                  value={editedQuestion?.metadata?.explanation || selectedQuestion.metadata?.explanation || ''}
                  onChange={(e) => setEditedQuestion({ ...editedQuestion, metadata: { ...editedQuestion?.metadata, explanation: e.target.value } })}
                  className="w-full p-6 bg-gradient-to-br from-indigo-50 to-blue-50 rounded-2xl border-2 border-indigo-200 text-gray-800 text-lg leading-relaxed"
                  rows={4}
                />
              </div>

              {/* Action Buttons */}
              <div className="flex justify-center space-x-4 pt-8">
                <button
                  onClick={() => handleApproval(selectedQuestion.id, 'approved')}
                  className="group relative px-8 py-4 bg-gradient-to-br from-green-600 via-green-700 to-emerald-600 text-white font-semibold rounded-2xl shadow-lg hover:shadow-xl transform transition-all duration-300 hover:scale-105 active:scale-95"
                >
                  <div className="flex items-center space-x-3">
                    <CheckCircle className="h-5 w-5 group-hover:rotate-12 transition-transform duration-300" />
                    <span className="text-lg">Approve Question</span>
                  </div>
                </button>
                <button
                  onClick={() => handleApproval(selectedQuestion.id, 'rejected')}
                  className="group relative px-8 py-4 bg-gradient-to-br from-red-600 via-red-700 to-pink-600 text-white font-semibold rounded-2xl shadow-lg hover:shadow-xl transform transition-all duration-300 hover:scale-105 active:scale-95"
                >
                  <div className="flex items-center space-x-3">
                    <XCircle className="h-5 w-5 group-hover:rotate-12 transition-transform duration-300" />
                    <span className="text-lg">Reject Question</span>
                  </div>
                </button>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50 to-indigo-100 p-6">
      <div className="max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between">
          <div>
            <h1 className="text-4xl font-bold text-gray-800 mb-2">Question Review Center</h1>
            <p className="text-gray-600 flex items-center">
              <Clock className="w-4 h-4 mr-2" />
              {stats.pending} questions awaiting review
            </p>
          </div>
          <div className="flex items-center space-x-4 mt-4 lg:mt-0">
            <div className="relative">
              <input
                type="text"
                placeholder="Search questions..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-4 pr-10 py-3 bg-white border-2 border-gray-200 rounded-2xl focus:border-blue-500 focus:ring-4 focus:ring-blue-100 transition-all duration-300"
              />
            </div>
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div className="group relative overflow-hidden bg-gradient-to-br from-blue-50 to-blue-100 rounded-2xl shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2 border border-blue-200/50">
            <div className="absolute top-0 right-0 w-32 h-32 bg-blue-500/10 rounded-full -translate-y-16 translate-x-16"></div>
            <div className="relative p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="p-3 bg-blue-500/20 backdrop-blur-sm rounded-xl border border-blue-300/30">
                  <Eye className="w-6 h-6 text-blue-600" />
                </div>
              </div>
              <h3 className="text-blue-700 text-sm font-semibold mb-1 uppercase tracking-wide">Pending Review</h3>
              <p className="text-3xl font-bold text-blue-900 mb-1">{stats.pending}</p>
            </div>
          </div>
          <div className="group relative overflow-hidden bg-gradient-to-br from-green-50 to-green-100 rounded-2xl shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2 border border-green-200/50">
            <div className="absolute top-0 right-0 w-32 h-32 bg-green-500/10 rounded-full -translate-y-16 translate-x-16"></div>
            <div className="relative p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="p-3 bg-green-500/20 backdrop-blur-sm rounded-xl border border-green-300/30">
                  <ThumbsUp className="w-6 h-6 text-green-600" />
                </div>
              </div>
              <h3 className="text-green-700 text-sm font-semibold mb-1 uppercase tracking-wide">Approved</h3>
              <p className="text-3xl font-bold text-green-900 mb-1">{stats.approved}</p>
            </div>
          </div>
          <div className="group relative overflow-hidden bg-gradient-to-br from-red-50 to-red-100 rounded-2xl shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2 border border-red-200/50">
            <div className="absolute top-0 right-0 w-32 h-32 bg-red-500/10 rounded-full -translate-y-16 translate-x-16"></div>
            <div className="relative p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="p-3 bg-red-500/20 backdrop-blur-sm rounded-xl border border-red-300/30">
                  <ThumbsDown className="w-6 h-6 text-red-600" />
                </div>
              </div>
              <h3 className="text-red-700 text-sm font-semibold mb-1 uppercase tracking-wide">Rejected</h3>
              <p className="text-3xl font-bold text-red-900 mb-1">{stats.rejected}</p>
            </div>
          </div>
        </div>

        {/* Questions List */}
        <div className="bg-white/95 backdrop-blur-sm rounded-3xl shadow-2xl p-8 border border-gray-200">
          <div className="space-y-6">
            {pendingQuestions.length === 0 ? (
              <div className="text-center py-12">
                <div className="w-24 h-24 bg-gradient-to-br from-green-500 to-emerald-600 rounded-full flex items-center justify-center mx-auto mb-4">
                  <CheckCircle className="w-12 h-12 text-white" />
                </div>
                <h3 className="text-xl font-semibold text-gray-800 mb-2">All Caught Up!</h3>
                <p className="text-gray-600">No questions pending review at the moment.</p>
              </div>
            ) : (
              pendingQuestions
                .filter(q => q.question.toLowerCase().includes(searchTerm.toLowerCase()))
                .map((question) => {
                  const TypeIcon = getTypeIcon(question.question_type);
                  return (
                    <div
                      key={question.id}
                      className="group relative overflow-hidden bg-gradient-to-r from-white to-gray-50 rounded-2xl shadow-lg hover:shadow-xl transition-all duration-300 transform hover:-translate-y-1 border-2 border-gray-100 hover:border-blue-200"
                    >
                      <div className="p-6">
                        <div className="flex items-start justify-between mb-4">
                          <div className="flex-1">
                            <div className="flex items-center space-x-3 mb-3">
                              <div className="p-2 bg-gradient-to-br from-blue-500 to-purple-600 rounded-lg">
                                <TypeIcon className="h-5 w-5 text-white" />
                              </div>
                              <div className={`px-3 py-1 bg-gradient-to-r ${getDifficultyColor(question.difficulty)} text-white rounded-full text-sm font-semibold`}>
                                {question.difficulty}
                              </div>
                              <div className="px-3 py-1 bg-gradient-to-r from-gray-500 to-slate-600 text-white rounded-full text-sm font-semibold">
                                {question.question_type.toUpperCase()}
                              </div>
                            </div>
                            <h3 className="text-lg font-semibold text-gray-800 mb-2 line-clamp-2">
                              {question.question}
                            </h3>
                            <div className="flex items-center space-x-6 text-sm text-gray-600">
                              <div className="flex items-center space-x-1">
                                <BookOpen className="h-4 w-4" />
                                <span>{question.subject.name} • {question.subtopic ? question.subtopic.name : 'N/A'}</span>
                              </div>
                              <div className="flex items-center space-x-1">
                                <Building className="h-4 w-4" />
                                <span>{question.school}</span>
                              </div>
                              <div className="flex items-center space-x-1">
                                <User className="h-4 w-4" />
                                <span>{question.user.name}</span>
                              </div>
                              <div className="flex items-center space-x-1">
                                <Calendar className="h-4 w-4" />
                                <span>{formatDate(question.created_at)}</span>
                              </div>
                            </div>
                          </div>
                          <div className="flex items-center space-x-3">
                            <button
                              onClick={() => {
                                setSelectedQuestion(question);
                                setEditedQuestion({
                                  ...question,
                                  subject: question.subject.name,
                                  subtopic: question.subtopic ? question.subtopic.name : null,
                                  answer: question.question_type === 'multiple_choice' ? JSON.parse(question.answer) : question.answer,
                                  question_type: question.question_type
                                });
                                setReviewMode('detail');
                              }}
                              className="group relative px-6 py-3 bg-gradient-to-br from-blue-600 via-blue-700 to-indigo-600 text-white font-semibold rounded-xl shadow-lg hover:shadow-xl transform transition-all duration-300 hover:scale-105 active:scale-95"
                            >
                              <div className="flex items-center space-x-2">
                                <Eye className="h-4 w-4 group-hover:rotate-12 transition-transform duration-300" />
                                <span>Review</span>
                              </div>
                            </button>
                          </div>
                        </div>
                      </div>
                    </div>
                  );
                })
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default AdminDashboard;