import React, { useState, useEffect, useMemo } from 'react';
import { 
  Edit, Save, X, AlertCircle, Loader, Clock, BookOpen, Award, Trash2,
  CheckCircle, XCircle, ArrowRight, ArrowLeft, RotateCcw, Play, Pause 
} from 'lucide-react';
import RequirePremium from './RequirePremium';

const PER_PAGE_OPTIONS = [12, 24, 36];

const Courses = () => {
  const [courses, setCourses] = useState([]);
  const [pagination, setPagination] = useState({
    current_page: 1,
    last_page: 1,
    total: 0,
    per_page: 12,
  });
  const [editingCourse, setEditingCourse] = useState(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [fetching, setFetching] = useState(true);
  const [selectedCourse, setSelectedCourse] = useState(null);
  const [showModeModal, setShowModeModal] = useState(false);
  const [showQuizConfig, setShowQuizConfig] = useState(false);
  const [currentMode, setCurrentMode] = useState(null);
  const [questions, setQuestions] = useState([]);
  const [currentQuestionIndex, setCurrentQuestionIndex] = useState(0);
  const [userAnswers, setUserAnswers] = useState({});
  const [showResults, setShowResults] = useState(false);
  const [quizDuration, setQuizDuration] = useState(15);
  const [timeLeft, setTimeLeft] = useState(0);
  const [isQuizActive, setIsQuizActive] = useState(false);
  const [showExplanation, setShowExplanation] = useState(false);

  const [currentPage, setCurrentPage] = useState(1);
  const [perPage, setPerPage] = useState(12);

  const API_URL = import.meta.env.VITE_API_URL || "http://localhost:8000";

  // Fetch courses with server-side pagination
  const fetchCourses = async (page = 1, perPageVal = perPage) => {
    setFetching(true);
    setError('');
    try {
      const url = new URL(`${API_URL}/api/courses`);
      url.searchParams.append('page', page);
      url.searchParams.append('per_page', perPageVal);

      const response = await fetch(url, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('auth_token') || ''}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || `Server error: ${response.status}`);
      }

      const data = await response.json();

      if (!data.success) {
        throw new Error(data.message || 'Failed to load courses');
      }

      setCourses(data.courses || []);
      setPagination({
        current_page: data.current_page,
        last_page: data.last_page,
        total: data.total,
        per_page: data.per_page,
      });
      setCurrentPage(data.current_page);
      setPerPage(data.per_page);
    } catch (err) {
      setError(err.message || 'Failed to fetch courses');
      setCourses([]);
    } finally {
      setFetching(false);
    }
  };

  // Load courses when page or per_page changes
  useEffect(() => {
    fetchCourses(currentPage, perPage);
  }, [currentPage, perPage]);

  const goToPage = (page) => {
    if (page >= 1 && page <= pagination.last_page && page !== currentPage) {
      setCurrentPage(page);
      window.scrollTo({ top: 0, behavior: 'smooth' });
    }
  };

  const handlePerPageChange = (newPerPage) => {
    setPerPage(newPerPage);
    setCurrentPage(1);
  };

  // Delete course and refetch current page
  const deleteCourse = async (course) => {
    if (!window.confirm(`Are you sure you want to delete "${course.name}"? This action cannot be undone.`)) {
      return;
    }

    setLoading(true);
    setError('');
    try {
      const response = await fetch(`${API_URL}/api/delete-course/${course.id}`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('auth_token') || ''}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || 'Failed to delete course');
      }

      // Refetch the current page to keep pagination accurate
      fetchCourses(currentPage, perPage);
    } catch (err) {
      setError(err.message || 'Failed to delete course');
    } finally {
      setLoading(false);
    }
  };

  // Timer effect
  useEffect(() => {
    let interval = null;
    if (isQuizActive && timeLeft > 0) {
      interval = setInterval(() => {
        setTimeLeft((time) => {
          if (time <= 1) {
            setIsQuizActive(false);
            setShowResults(true);
            clearInterval(interval);
            return 0;
          }
          return time - 1;
        });
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [isQuizActive, timeLeft]);

  const fetchQuestions = async (courseId) => {
    setLoading(true);
    setError('');
    try {
      const response = await fetch(`${API_URL}/api/personal_questions?course_id=${courseId}`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('auth_token') || ''}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || 'Failed to fetch questions');
      }

      const data = await response.json();
      setQuestions(Array.isArray(data) ? data : []);
      return Array.isArray(data) ? data : [];
    } catch (err) {
      setError(err.message || 'Failed to fetch questions');
      setQuestions([]);
      return [];
    } finally {
      setLoading(false);
    }
  };

  const startEditing = (course) => {
    setEditingCourse({ ...course });
  };

  const handleEditChange = (e) => {
    const { name, value } = e.target;
    setEditingCourse((prev) => ({ ...prev, [name]: value }));
  };

  const handleSaveEdit = async (courseId) => {
    if (!editingCourse?.name?.trim()) {
      setError('Course name cannot be empty');
      return;
    }

    setLoading(true);
    try {
      const response = await fetch(`${API_URL}/api/edit-course/${courseId}`, {
        method: 'PUT',
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('auth_token') || ''}`,
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        body: JSON.stringify({ name: editingCourse.name }),
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || 'Failed to update course');
      }

      const updated = await response.json();

      setCourses((prev) =>
        prev.map((c) => (c.id === courseId ? { ...c, name: updated.name } : c))
      );
      setEditingCourse(null);
      setError('');
    } catch (err) {
      setError(err.message || 'Failed to update course');
    } finally {
      setLoading(false);
    }
  };

  const cancelEditing = () => {
    setEditingCourse(null);
    setError('');
  };

  const handleCourseClick = (course) => {
    setSelectedCourse({ ...course });
    setShowModeModal(true);
  };

  const handleModeSelect = async (mode) => {
    setCurrentMode(mode);
    setShowModeModal(false);

    if (mode === 'quiz') {
      setShowQuizConfig(true);
    } else {
      await startLearningMode();
    }
  };

  const startLearningMode = async () => {
    const data = await fetchQuestions(selectedCourse.id);
    if (data.length === 0) {
      setError('No questions available for this course');
      return;
    }
    setCurrentQuestionIndex(0);
    setUserAnswers({});
    setShowExplanation(false);
  };

  const startQuizMode = async () => {
    const data = await fetchQuestions(selectedCourse.id);
    if (data.length === 0) {
      setError('No questions available for this course');
      setShowQuizConfig(false);
      return;
    }
    setTimeLeft(quizDuration * 60);
    setIsQuizActive(true);
    setCurrentQuestionIndex(0);
    setUserAnswers({});
    setShowResults(false);
    setShowQuizConfig(false);
  };

  const handleAnswerSelect = (questionId, selectedAnswer) => {
    setUserAnswers((prev) => ({
      ...prev,
      [questionId]: selectedAnswer,
    }));

    if (currentMode === 'learning') {
      setShowExplanation(true);
    }
  };

  const nextQuestion = () => {
    if (currentQuestionIndex < questions.length - 1) {
      setCurrentQuestionIndex((prev) => prev + 1);
      setShowExplanation(false);
    } else if (currentMode === 'quiz') {
      setIsQuizActive(false);
      setShowResults(true);
    } else if (currentMode === 'learning') {
      resetAll();
    } else if (currentMode === 'review') {
      setShowResults(true);
    }
  };

  const previousQuestion = () => {
    if (currentQuestionIndex > 0) {
      setCurrentQuestionIndex((prev) => prev - 1);
      setShowExplanation(currentMode === 'learning' && userAnswers[questions[currentQuestionIndex - 1]?.id]);
    }
  };

  const calculateResults = useMemo(() => {
    if (!showResults) return null;
    const totalQuestions = questions.length;
    const answeredQuestions = Object.keys(userAnswers).length;
    const correctAnswers = questions.filter((q) => userAnswers[q.id] === q.answer).length;
    const percentage = totalQuestions > 0 ? Math.round((correctAnswers / totalQuestions) * 100) : 0;

    return {
      totalQuestions,
      answeredQuestions,
      correctAnswers,
      wrongAnswers: answeredQuestions - correctAnswers,
      unanswered: totalQuestions - answeredQuestions,
      percentage,
    };
  }, [showResults, questions, userAnswers]);

  const resetAll = () => {
    setSelectedCourse(null);
    setCurrentMode(null);
    setQuestions([]);
    setCurrentQuestionIndex(0);
    setUserAnswers({});
    setShowResults(false);
    setShowQuizConfig(false);
    setShowModeModal(false);
    setTimeLeft(0);
    setIsQuizActive(false);
    setShowExplanation(false);
    setError('');
  };

  const formatTime = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  const currentQuestion = questions[currentQuestionIndex];

  // Results Screen
  if (showResults) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-100 via-white to-purple-100 font-sans text-gray-800">
        <div className="max-w-4xl mx-auto p-6 lg:p-12">
          <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-8 sm:p-12 text-center">
            <div className="mb-8">
              <div className="w-20 h-20 bg-amber-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Award className="w-12 h-12 text-amber-600" />
              </div>
              <h1 className="text-4xl font-bold text-gray-900 mb-2">Quiz Complete!</h1>
              <p className="text-gray-600 text-lg">{selectedCourse?.name}</p>
            </div>

            <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-8">
              <div className="bg-emerald-50 border border-emerald-200 p-6 rounded-xl">
                <CheckCircle className="w-8 h-8 text-emerald-600 mx-auto mb-2" />
                <div className="text-2xl font-bold text-emerald-600">{calculateResults?.correctAnswers || 0}</div>
                <div className="text-sm text-emerald-700 font-medium">Correct</div>
              </div>
              <div className="bg-red-50 border border-red-200 p-6 rounded-xl">
                <XCircle className="w-8 h-8 text-red-600 mx-auto mb-2" />
                <div className="text-2xl font-bold text-red-600">{calculateResults?.wrongAnswers || 0}</div>
                <div className="text-sm text-red-700 font-medium">Wrong</div>
              </div>
              <div className="bg-gray-50 border border-gray-200 p-6 rounded-xl">
                <Clock className="w-8 h-8 text-gray-600 mx-auto mb-2" />
                <div className="text-2xl font-bold text-gray-600">{calculateResults?.unanswered || 0}</div>
                <div className="text-sm text-gray-700 font-medium">Unanswered</div>
              </div>
              <div className="bg-indigo-50 border border-indigo-200 p-6 rounded-xl">
                <Award className="w-8 h-8 text-indigo-600 mx-auto mb-2" />
                <div className="text-2xl font-bold text-indigo-600">{calculateResults?.percentage || 0}%</div>
                <div className="text-sm text-indigo-700 font-medium">Score</div>
              </div>
            </div>

            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <button
                onClick={() => {
                  setShowResults(false);
                  setCurrentQuestionIndex(0);
                  setIsQuizActive(false);
                  setShowExplanation(true);
                  setCurrentMode('review');
                }}
                className="bg-indigo-600 text-white py-3 px-8 rounded-lg font-semibold hover:bg-indigo-700 transition-all duration-200 flex items-center justify-center space-x-2 shadow-sm"
              >
                <RotateCcw className="w-5 h-5" />
                <span>Review Answers</span>
              </button>
              <button
                onClick={resetAll}
                className="bg-white border border-gray-300 text-gray-700 py-3 px-8 rounded-lg font-semibold hover:bg-gray-50 transition-all duration-200"
              >
                Back to Courses
              </button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Question Screen (Learning / Quiz / Review)
  if (currentQuestion && selectedCourse && (currentMode === 'learning' || currentMode === 'quiz' || currentMode === 'review')) {
    const isReviewMode = currentMode === 'review';

    return (
      <div className="min-h-screen bg-gray-50 font-sans text-gray-800">
        <div className="max-w-4xl mx-auto p-6 lg:p-12">
          {/* Header */}
          <div className="flex items-center justify-between mb-8">
            <div className="flex items-center space-x-4">
              <button onClick={resetAll} className="text-gray-600 hover:text-gray-900">
                <ArrowLeft className="w-6 h-6" />
              </button>
              <div>
                <h1 className="text-2xl font-bold text-gray-900">{selectedCourse.name}</h1>
                <p className="text-gray-600 capitalize flex items-center space-x-2">
                  {currentMode === 'learning' ? <BookOpen className="w-4 h-4" /> : <Award className="w-4 h-4" />}
                  <span>{isReviewMode ? 'Review' : currentMode} Mode</span>
                </p>
              </div>
            </div>

            {currentMode === 'quiz' && !isReviewMode && (
              <div className="flex items-center space-x-4">
                <button
                  onClick={() => setIsQuizActive(!isQuizActive)}
                  className="bg-white border border-gray-300 text-gray-700 p-2 rounded-lg hover:bg-gray-50"
                >
                  {isQuizActive ? <Pause className="w-5 h-5" /> : <Play className="w-5 h-5" />}
                </button>
                <div className={`bg-white border border-gray-300 px-4 py-2 rounded-lg font-mono text-lg font-bold ${timeLeft <= 60 ? 'text-red-600' : 'text-gray-900'}`}>
                  <Clock className="w-5 h-5 inline mr-2" />
                  {formatTime(timeLeft)}
                </div>
              </div>
            )}
          </div>

          {/* Progress Bar */}
          <div className="mb-8">
            <div className="flex justify-between items-center mb-2">
              <span className="text-sm font-medium text-gray-700">
                Question {currentQuestionIndex + 1} of {questions.length}
              </span>
              <span className="text-sm font-medium text-gray-700">
                {Math.round(((currentQuestionIndex + 1) / questions.length) * 100)}% Complete
              </span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div
                className="bg-indigo-600 h-2 rounded-full transition-all duration-300"
                style={{ width: `${((currentQuestionIndex + 1) / questions.length) * 100}%` }}
              />
            </div>
          </div>

          {/* Question Card */}
          <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-8 sm:p-12 mb-8">
            <h2 className="text-xl sm:text-2xl font-semibold text-gray-900 mb-8 leading-relaxed">
              {currentQuestion.question}
            </h2>

            <div className="space-y-4 mb-8">
              {Array.isArray(currentQuestion.options) ? (
                currentQuestion.options.map((option, index) => {
                  const isSelected = userAnswers[currentQuestion.id] === option;
                  const isCorrect = option === currentQuestion.answer;
                  const showCorrectAnswer = isReviewMode || (showExplanation && currentMode === 'learning');

                  let buttonClass = 'w-full p-4 text-left rounded-lg border-2 transition-all duration-200 ';

                  if (showCorrectAnswer) {
                    if (isCorrect) buttonClass += 'bg-emerald-50 border-emerald-500 text-emerald-900';
                    else if (isSelected && !isCorrect) buttonClass += 'bg-red-50 border-red-500 text-red-900';
                    else buttonClass += 'bg-gray-50 border-gray-300 text-gray-600';
                  } else if (isSelected) {
                    buttonClass += 'bg-indigo-50 border-indigo-500 text-indigo-900';
                  } else {
                    buttonClass += 'bg-white border-gray-300 text-gray-900 hover:bg-gray-50 hover:border-gray-400';
                  }

                  return (
                    <button
                      key={`${currentQuestion.id}-${option}`}
                      onClick={() => !isReviewMode && handleAnswerSelect(currentQuestion.id, option)}
                      disabled={isReviewMode || (showExplanation && currentMode === 'learning')}
                      className={buttonClass}
                    >
                      <div className="flex items-center space-x-3">
                        <div className={`w-6 h-6 rounded-full border-2 flex items-center justify-center text-xs font-bold ${
                          showCorrectAnswer && isCorrect ? 'bg-emerald-500 border-emerald-500 text-white' :
                          showCorrectAnswer && isSelected && !isCorrect ? 'bg-red-500 border-red-500 text-white' :
                          isSelected ? 'bg-indigo-500 border-indigo-500 text-white' :
                          'border-gray-300 text-gray-500'
                        }`}>
                          {String.fromCharCode(65 + index)}
                        </div>
                        <span className="flex-1">{option}</span>
                        {showCorrectAnswer && isCorrect && <CheckCircle className="w-5 h-5 text-emerald-500" />}
                        {showCorrectAnswer && isSelected && !isCorrect && <XCircle className="w-5 h-5 text-red-500" />}
                      </div>
                    </button>
                  );
                })
              ) : (
                <p className="text-red-600">No options available for this question.</p>
              )}
            </div>

            {(isReviewMode || (showExplanation && currentMode === 'learning')) && currentQuestion.explanation && (
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-6">
                <div className="flex items-start space-x-3">
                  <AlertCircle className="w-5 h-5 text-blue-600 flex-shrink-0 mt-0.5" />
                  <div>
                    <h3 className="font-semibold text-blue-900 mb-2">Explanation</h3>
                    <p className="text-blue-800 leading-relaxed">{currentQuestion.explanation}</p>
                  </div>
                </div>
              </div>
            )}
          </div>

          {/* Navigation Buttons */}
          <div className="flex justify-between items-center">
            <button
              onClick={previousQuestion}
              disabled={currentQuestionIndex === 0}
              className="flex items-center space-x-2 bg-white border border-gray-300 text-gray-700 py-3 px-6 rounded-lg font-semibold hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              <ArrowLeft className="w-5 h-5" />
              <span>Previous</span>
            </button>

            <div className="flex items-center space-x-4">
              {currentMode === 'learning' && userAnswers[currentQuestion.id] && !isReviewMode && (
                <div className="text-sm font-semibold">
                  {userAnswers[currentQuestion.id] === currentQuestion.answer ? (
                    <span className="text-emerald-600">Correct!</span>
                  ) : (
                    <span className="text-red-600">Incorrect</span>
                  )}
                </div>
              )}

              <button
                onClick={nextQuestion}
                disabled={!isReviewMode && !userAnswers[currentQuestion.id] && currentMode !== 'review'}
                className="flex items-center space-x-2 bg-indigo-600 text-white py-3 px-6 rounded-lg font-semibold hover:bg-indigo-700 disabled:opacity-50 disabled:cursor-not-allowed shadow-sm"
              >
                <span>
                  {currentQuestionIndex === questions.length - 1
                    ? isReviewMode ? 'Back to Results' : currentMode === 'quiz' ? 'Finish Quiz' : 'Complete'
                    : 'Next'}
                </span>
                <ArrowRight className="w-5 h-5" />
              </button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Error Screen
  if (error && !currentQuestion && !showResults) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-100 via-white to-purple-100 flex items-center justify-center">
        <div className="bg-white/90 backdrop-blur-md rounded-3xl shadow-2xl border border-white/30 p-8 sm:p-12 text-center max-w-md">
          <AlertCircle className="w-12 h-12 text-red-500 mx-auto mb-4" />
          <h2 className="text-2xl font-bold text-gray-800 mb-4">Error</h2>
          <p className="text-gray-600 mb-6">{error}</p>
          <button
            onClick={resetAll}
            className="bg-gradient-to-r from-indigo-600 to-purple-600 text-white py-3 px-8 rounded-2xl font-semibold hover:from-indigo-700 hover:to-purple-700 transition-all duration-300"
          >
            Back to Courses
          </button>
        </div>
      </div>
    );
  }

  // Loading Questions
  if (loading && selectedCourse && currentMode && !showResults) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-indigo-100 via-white to-purple-100 flex items-center justify-center">
        <div className="text-center">
          <Loader className="w-12 h-12 text-indigo-600 animate-spin mx-auto mb-4" />
          <p className="text-gray-600 text-lg">Loading questions...</p>
        </div>
      </div>
    );
  }

  // MAIN COURSES LIST WITH SERVER-SIDE PAGINATION
  return (
    <RequirePremium>
    <div className="min-h-screen bg-gray-50 font-sans text-gray-800">
      <div className="max-w-5xl mx-auto p-0 lg:p-12">
        <div className="text-center mb-12">
          <h1 className="text-4xl sm:text-5xl font-bold text-gray-900 mb-4">
            Explore Your Quiz
          </h1>
          <p className="text-gray-600 text-lg sm:text-xl max-w-2xl mx-auto">
            Choose between Learning Mode or Quiz Mode to start your journey
          </p>
        </div>

        <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-8 sm:p-12">
          <div className="flex justify-between items-center mb-8">
            <h2 className="text-2xl font-semibold text-gray-900">Your Courses</h2>
            {pagination.total > 0 && (
              <div className="text-sm text-gray-600">
                {pagination.total} course{pagination.total !== 1 ? 's' : ''}
              </div>
            )}
          </div>

          {fetching ? (
            <div className="flex justify-center items-center py-12">
              <Loader className="w-8 h-8 text-indigo-600 animate-spin" />
              <span className="ml-3 text-gray-600 text-lg">Loading courses...</span>
            </div>
          ) : error ? (
            <div className="bg-red-50 border border-red-200 rounded-lg p-4 flex items-center space-x-3">
              <AlertCircle className="w-5 h-5 text-red-600 flex-shrink-0" />
              <span className="text-red-700 text-sm">{error}</span>
            </div>
          ) : courses.length === 0 ? (
            <p className="text-gray-600 text-center text-lg py-12">
              No courses found. Please upload PDF to the AI Quiz Generator
            </p>
          ) : (
            <>
              {/* Courses Grid */}
              <div className="grid gap-6 sm:grid-cols-2 lg:grid-cols-3 mb-12">
                {courses.map((course) => (
                  <div
                    key={course.id}
                    className="border border-gray-200 rounded-xl p-6 bg-white shadow-sm hover:shadow-md transition-all duration-200 hover:border-gray-300"
                  >
                    {editingCourse && editingCourse.id === course.id ? (
                      <div className="space-y-4">
                        <input
                          type="text"
                          name="name"
                          value={editingCourse.name}
                          onChange={handleEditChange}
                          className="w-full p-3 border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none"
                          placeholder="Enter course name"
                          disabled={loading}
                          autoFocus
                        />
                        {error && editingCourse.id === course.id && (
                          <div className="bg-red-50 border border-red-200 rounded-lg p-3 flex items-center space-x-2">
                            <AlertCircle className="w-5 h-5 text-red-600 flex-shrink-0" />
                            <span className="text-red-700 text-sm">{error}</span>
                          </div>
                        )}
                        <div className="flex space-x-3">
                          <button
                            onClick={() => handleSaveEdit(course.id)}
                            disabled={loading}
                            className="flex-1 bg-indigo-600 text-white py-2 px-4 rounded-lg font-semibold hover:bg-indigo-700 disabled:opacity-50 flex items-center justify-center space-x-2"
                          >
                            <Save className="w-4 h-4" />
                            <span>Save</span>
                          </button>
                          <button
                            onClick={cancelEditing}
                            disabled={loading}
                            className="flex-1 bg-white border border-gray-300 text-gray-700 py-2 px-4 rounded-lg font-semibold hover:bg-gray-50 flex items-center justify-center space-x-2"
                          >
                            <X className="w-4 h-4" />
                            <span>Cancel</span>
                          </button>
                        </div>
                      </div>
                    ) : (
                      <div className="flex justify-between items-start">
                        <div
                          onClick={() => handleCourseClick(course)}
                          className="cursor-pointer flex-1 pr-4"
                        >
                          <h3 className="text-lg font-semibold text-gray-900 line-clamp-2 mb-2">
                            {course.name}
                          </h3>
                          {(course.subject || course.subtopic) && (
                            <div className="text-sm text-gray-600 space-y-1 mb-3">
                              {course.subject && <p>{course.subject}</p>}
                              {course.subtopic && <p>{course.subtopic}</p>}
                            </div>
                          )}
                          <div className="text-xs text-gray-500">
                            {course.personal_questions_count} question{course.personal_questions_count !== 1 ? 's' : ''}
                          </div>
                        </div>
                        <div className="flex space-x-2 flex-shrink-0">
                          <button
                            onClick={(e) => { e.stopPropagation(); startEditing(course); }}
                            className="text-gray-500 hover:text-indigo-600 p-1"
                          >
                            <Edit className="w-5 h-5" />
                          </button>
                          <button
                            onClick={(e) => { e.stopPropagation(); deleteCourse(course); }}
                            className="text-gray-500 hover:text-red-600 p-1"
                          >
                            <Trash2 className="w-5 h-5" />
                          </button>
                        </div>
                      </div>
                    )}
                  </div>
                ))}
              </div>

              {/* Server-Side Pagination */}
              {pagination.last_page > 1 && (
                <div className="flex flex-col sm:flex-row items-center justify-between gap-6 py-6 border-t border-gray-200">
                  <div className="text-sm text-gray-600">
                    Showing <span className="font-semibold">
                      {(pagination.current_page - 1) * pagination.per_page + 1}
                    </span> to{' '}
                    <span className="font-semibold">
                      {Math.min(pagination.current_page * pagination.per_page, pagination.total)}
                    </span> of <span className="font-semibold">{pagination.total}</span> courses
                  </div>

                  <div className="flex items-center space-x-4">
                    {/* <div className="flex items-center space-x-2">
                      <span className="text-sm text-gray-600">Show:</span>
                      <select
                        value={perPage}
                        onChange={(e) => handlePerPageChange(Number(e.target.value))}
                        className="text-sm border border-gray-300 rounded-lg px-3 py-1.5 focus:border-indigo-500 focus:ring-1 focus:ring-indigo-500"
                      >
                        {PER_PAGE_OPTIONS.map((num) => (
                          <option key={num} value={num}>{num}</option>
                        ))}
                      </select>
                    </div> */}

                    <div className="flex items-center space-x-2">
                      <button
                        onClick={() => goToPage(currentPage - 1)}
                        disabled={currentPage === 1}
                        className="px-4 py-2 rounded-lg border border-gray-300 text-sm font-medium bg-white text-gray-700 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                      >
                        Previous
                      </button>

                      {Array.from({ length: Math.min(pagination.last_page, 5) }, (_, i) => {
                        let pageNum = i + 1;
                        if (pagination.last_page > 5) {
                          if (currentPage > 3 && currentPage < pagination.last_page - 2) {
                            pageNum = currentPage - 2 + i;
                          } else if (currentPage >= pagination.last_page - 2) {
                            pageNum = pagination.last_page - 4 + i;
                          }
                        }
                        return (
                          <button
                            key={pageNum}
                            onClick={() => goToPage(pageNum)}
                            className={`px-4 py-2 rounded-lg text-sm font-medium transition-all ${
                              currentPage === pageNum
                                ? 'bg-indigo-600 text-white'
                                : 'bg-white border border-gray-300 text-gray-700 hover:bg-gray-50'
                            }`}
                          >
                            {pageNum}
                          </button>
                        );
                      })}

                      {pagination.last_page > 5 && currentPage < pagination.last_page - 2 && (
                        <>
                          <span className="text-gray-500 px-2">...</span>
                          <button
                            onClick={() => goToPage(pagination.last_page)}
                            className="px-4 py-2 rounded-lg border border-gray-300 text-sm font-medium bg-white text-gray-700 hover:bg-gray-50"
                          >
                            {pagination.last_page}
                          </button>
                        </>
                      )}

                      <button
                        onClick={() => goToPage(currentPage + 1)}
                        disabled={currentPage === pagination.last_page}
                        className="px-4 py-2 rounded-lg border border-gray-300 text-sm font-medium bg-white text-gray-700 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                      >
                        Next
                      </button>
                    </div>
                  </div>
                </div>
              )}
            </>
          )}
        </div>

        {/* Mode Selection Modal */}
        {showModeModal && selectedCourse && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
            <div className="bg-white rounded-2xl shadow-xl border border-gray-200 p-8 sm:p-12 max-w-md w-full">
              <div className="flex justify-between items-center mb-6">
                <h2 className="text-2xl font-bold text-gray-900">{selectedCourse.name}</h2>
                <button onClick={() => setShowModeModal(false)} className="text-gray-500 hover:text-gray-700">
                  <X className="w-6 h-6" />
                </button>
              </div>
              <p className="text-gray-600 mb-6">Choose a mode to start:</p>
              <div className="space-y-4">
                <button
                  onClick={() => handleModeSelect('learning')}
                  className="w-full bg-indigo-600 text-white py-3 px-6 rounded-lg font-semibold hover:bg-indigo-700 flex items-center justify-center space-x-2 shadow-sm"
                >
                  <BookOpen className="w-5 h-5" />
                  <span>Learning Mode</span>
                </button>
                <button
                  onClick={() => handleModeSelect('quiz')}
                  className="w-full bg-white border border-gray-300 text-gray-700 py-3 px-6 rounded-lg font-semibold hover:bg-gray-50 flex items-center justify-center space-x-2"
                >
                  <Award className="w-5 h-5" />
                  <span>Quiz Mode</span>
                </button>
              </div>
            </div>
          </div>
        )}

        {/* Quiz Configuration Modal */}
        {showQuizConfig && selectedCourse && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
            <div className="bg-white rounded-2xl shadow-xl border border-gray-200 p-8 sm:p-12 max-w-md w-full">
              <div className="flex justify-between items-center mb-6">
                <h2 className="text-2xl font-bold text-gray-900">Quiz Configuration</h2>
                <button onClick={() => setShowQuizConfig(false)} className="text-gray-500 hover:text-gray-700">
                  <X className="w-6 h-6" />
                </button>
              </div>
              <p className="text-gray-600 mb-6">Set the duration for the quiz (in minutes):</p>
              <div className="mb-6">
                <input
                  type="number"
                  value={quizDuration}
                  onChange={(e) => setQuizDuration(Math.max(1, parseInt(e.target.value) || 15))}
                  className="w-full p-3 border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none"
                  min="1"
                />
              </div>
              <div className="flex space-x-4">
                <button
                  onClick={startQuizMode}
                  className="flex-1 bg-indigo-600 text-white py-3 px-6 rounded-lg font-semibold hover:bg-indigo-700 flex items-center justify-center space-x-2 shadow-sm"
                >
                  <Play className="w-5 h-5" />
                  <span>Start Quiz</span>
                </button>
                <button
                  onClick={() => {
                    setShowQuizConfig(false);
                    setSelectedCourse(null);
                    setCurrentMode(null);
                  }}
                  className="flex-1 bg-white border border-gray-300 text-gray-700 py-3 px-6 rounded-lg font-semibold hover:bg-gray-50 flex items-center justify-center space-x-2"
                >
                  <X className="w-5 h-5" />
                  <span>Cancel</span>
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
    </RequirePremium>
  );
};

export default Courses;