import React, { useState } from 'react';
import {
  Calendar,
  Clock,
  BookOpen,
  Target,
  TrendingUp,
  Award,
  BarChart3,
  Activity,
  Brain,
  Timer,
} from 'lucide-react';
import { useStats } from './Dashboard'; // Adjust path as needed

const Main = () => {
  const { stats, recentActivity, loading } = useStats();
  const [selectedTimeframe, setSelectedTimeframe] = useState('week');

  // Safely calculate average accuracy with default 0
  const calculateAverageAccuracy = () => {
    if (loading.stats || !recentActivity || recentActivity.length === 0) {
      return 0;
    }
    const totalAccuracy = recentActivity.reduce((sum, activity) => sum + (activity.accuracy || 0), 0);
    return Math.round(totalAccuracy / recentActivity.length);
  };

  const accuracy = calculateAverageAccuracy();

  const examData = {
    totalQuestions: 10000,
    questionsAnswered: stats.questionsAnswered || 0,
    studyStreak: stats.studyStreak || 0,
    hoursStudied: 147,
    upcomingExams: [
      { name: 'OSCE Cardiology', date: '2025-07-25', timeLeft: '5 days' },
      { name: 'Internal Medicine MCQ', date: '2025-08-02', timeLeft: '13 days' },
      { name: 'Clinical Skills Assessment', date: '2025-08-15', timeLeft: '26 days' },
    ],
    subjectProgress: [
      { name: 'Cardiology', completed: 85, total: 120, percentage: 71 },
      { name: 'Neurology', completed: 67, total: 95, percentage: 71 },
      { name: 'Respiratory', completed: 92, total: 110, percentage: 84 },
      { name: 'Gastroenterology', completed: 45, total: 88, percentage: 51 },
      { name: 'Endocrinology', completed: 34, total: 76, percentage: 45 },
      { name: 'Nephrology', completed: 28, total: 64, percentage: 44 },
    ],
    weeklyGoals: {
      questionsTarget: 200,
      questionsCurrent: 147,
      studyHoursTarget: 25,
      studyHoursCurrent: 18.5,
      subjectsTarget: 5,
      subjectsCurrent: 3,
    },
    achievements: [
      { name: 'Cardiology Master', icon: 'Heart', earned: true },
      { name: '7-Day Streak', icon: 'Fire', earned: true },
      { name: 'Perfect Score', icon: '100', earned: false },
      { name: 'Speed Demon', icon: 'Lightning', earned: true },
    ],
  };

  const StatsCard = ({ title, value, subtitle, icon: Icon, color, trend }) => (
    <div className="bg-white rounded-xl shadow-lg p-6 border-l-4" style={{ borderLeftColor: color }}>
      <div className="flex items-center justify-between">
        <div>
          <p className="text-gray-600 text-sm font-medium mb-1">{title}</p>
          <p className="text-3xl font-bold text-gray-800">{value}</p>
          {subtitle && <p className="text-sm text-gray-500 mt-1">{subtitle}</p>}
          {trend && (
            <div className={`flex items-center mt-2 text-sm ${trend > 0 ? 'text-green-600' : 'text-red-600'}`}>
              <TrendingUp className="w-4 h-4 mr-1" />
              {trend > 0 ? '+' : ''}{trend}%
            </div>
          )}
        </div>
        <div className="p-3 rounded-full" style={{ backgroundColor: color + '20' }}>
          <Icon className="w-8 h-8" style={{ color }} />
        </div>
      </div>
    </div>
  );

  const ProgressBar = ({ percentage, color = '#3B82F6' }) => (
    <div className="w-full bg-gray-200 rounded-full h-2">
      <div
        className="h-2 rounded-full transition-all duration-300"
        style={{ width: `${percentage}%`, backgroundColor: color }}
      ></div>
    </div>
  );

  return (
    <div className="min-h-screen bg-gray-50 p-0">
      <div className="max-w-7xl mx-auto space-y-6">
        {/* Header */}
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between">
          <div>
            <h1 className="text-4xl font-bold text-gray-900 mb-2">User Dashboard</h1>
            <p className="text-gray-600 flex items-center">
              <Calendar className="w-4 h-4 mr-2" />
              {new Date().toLocaleDateString('en-US', {
                weekday: 'long',
                year: 'numeric',
                month: 'long',
                day: 'numeric',
              })}
            </p>
          </div>
        </div>

        {/* Key Performance Metrics */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {/* Questions Answered Card */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow duration-200">
            <div className="p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="p-3 bg-blue-50 rounded-xl border border-blue-100">
                  <Brain className="w-6 h-6 text-blue-600" />
                </div>
                <div className="flex items-center text-green-600 text-sm font-semibold bg-green-50 px-2 py-1 rounded-full">
                  <TrendingUp className="w-3 h-3 mr-1" />
                  +12%
                </div>
              </div>
              <h3 className="text-gray-600 text-sm font-semibold mb-1 uppercase tracking-wide">Questions Answered</h3>
              {loading.stats ? (
                <div className="animate-pulse h-9 w-20 bg-gray-200 rounded"></div>
              ) : (
                <>
                  <p className="text-3xl font-bold text-gray-900 mb-1">{examData.questionsAnswered.toLocaleString()}</p>
                  <p className="text-gray-600 text-sm">of {examData.totalQuestions.toLocaleString()} total</p>
                  <div className="mt-4">
                    <ProgressBar percentage={(examData.questionsAnswered / examData.totalQuestions) * 100} color="#3B82F6" />
                  </div>
                </>
              )}
            </div>
          </div>

          {/* Accuracy Rate Card */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow duration-200">
            <div className="p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="p-3 bg-emerald-50 rounded-xl border border-emerald-100">
                  <Target className="w-6 h-6 text-emerald-600" />
                </div>
                <div className="flex items-center text-green-600 text-sm font-semibold bg-green-50 px-2 py-1 rounded-full">
                  <TrendingUp className="w-3 h-3 mr-1" />
                  +5%
                </div>
              </div>
              <h3 className="text-gray-600 text-sm font-semibold mb-1 uppercase tracking-wide">Accuracy Rate</h3>
              {loading.stats || recentActivity?.length === 0 ? (
                <>
                  <p className="text-3xl font-bold text-gray-900 mb-1">0%</p>
                  <p className="text-gray-600 text-sm">No quiz data yet</p>
                  <div className="mt-4">
                    <ProgressBar percentage={0} color="#10B981" />
                  </div>
                </>
              ) : (
                <>
                  <p className="text-3xl font-bold text-gray-900 mb-1">{accuracy}%</p>
                  <p className="text-gray-600 text-sm">Based on recent performance</p>
                  <div className="mt-4 flex items-center gap-3">
                    <div className="flex-1">
                      <ProgressBar percentage={accuracy} color="#10B981" />
                    </div>
                    <span className="text-xs font-medium text-gray-700">
                      {accuracy >= 80 ? 'Excellent' : accuracy >= 70 ? 'Good' : 'Room to improve'}
                    </span>
                  </div>
                </>
              )}
            </div>
          </div>

          {/* Study Streak Card */}
          <div className="bg-white rounded-xl shadow-sm border border-gray-200 hover:shadow-md transition-shadow duration-200">
            <div className="p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="p-3 bg-amber-50 rounded-xl border border-amber-100">
                  <Award className="w-6 h-6 text-amber-600" />
                </div>
              </div>
              <h3 className="text-gray-600 text-sm font-semibold mb-1 uppercase tracking-wide">Study Streak</h3>
              {loading.stats ? (
                <div className="animate-pulse h-9 w-20 bg-gray-200 rounded"></div>
              ) : (
                <>
                  <div className="flex items-baseline mb-2">
                    <p className="text-3xl font-bold text-gray-900">{examData.studyStreak}</p>
                    <span className="text-gray-600 text-lg font-medium ml-2">days</span>
                  </div>
                  <p className="text-gray-600 text-sm flex items-center gap-2">
                    {/* <span className="text-2xl">Fire</span> */}
                    {examData.studyStreak > 0 ? 'Keep the streak alive!' : 'Start studying today!'}
                  </p>
                </>
              )}
            </div>
          </div>
        </div>

        {/* Recent Activity */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
          <h3 className="text-xl font-semibold text-gray-900 mb-4 flex items-center">
            <Activity className="w-5 h-5 mr-2 text-indigo-600" />
            Recent Activity
          </h3>
          <div className="space-y-4">
            {loading.stats ? (
              <div className="text-center py-8">
                <div className="animate-spin rounded-full h-10 w-10 border-b-2 border-indigo-600 mx-auto mb-4"></div>
                <p className="text-gray-500">Loading your recent sessions...</p>
              </div>
            ) : recentActivity?.length === 0 ? (
              <div className="text-center py-8 text-gray-500">
                <p className="text-lg mb-2">No recent quiz activity</p>
                <p className="text-sm">Your progress will appear here once you start studying</p>
              </div>
            ) : (
              recentActivity.map((activity) => (
                <div key={activity.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg border border-gray-200">
                  <div>
                    <h4 className="font-semibold text-gray-900">{activity.subject || 'General Quiz'}</h4>
                    <p className="text-sm text-gray-600 mt-1">
                      {activity.total_questions} questions • {activity.date}
                    </p>
                  </div>
                  <div
                    className={`px-4 py-2 rounded-full text-sm font-bold ${
                      activity.accuracy >= 80
                        ? 'bg-green-100 text-green-800'
                        : activity.accuracy >= 70
                        ? 'bg-yellow-100 text-yellow-800'
                        : 'bg-red-100 text-red-800'
                    }`}
                  >
                    {activity.accuracy}%
                  </div>
                </div>
              ))
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default Main;