import { useState, useEffect } from 'react';
import { Check, X, Sparkles, Crown, Zap, AlertCircle } from 'lucide-react';
import { Navigate, useNavigate } from 'react-router-dom';

function Paystack() {
  const navigate = useNavigate();
  const [email, setEmail] = useState('');
  const [selectedPlan, setSelectedPlan] = useState('1_month');
  const [error, setError] = useState('');
  const [loading, setLoading] = useState(false);
  const [paystackLoaded, setPaystackLoaded] = useState(false);
  const [saving, setSaving] = useState(false);
  const [showSuccess, setShowSuccess] = useState(false);
  const [successData, setSuccessData] = useState(null);
  const [subscriptionStatus, setSubscriptionStatus] = useState(null);
  const [checkingStatus, setCheckingStatus] = useState(true);

  // Subscription plans with pricing
  const subscriptionPlans = {
    '1_month': {
      duration: 1,
      price: 1500,
      originalPrice: 1500,
      discount: 0,
      label: '1 Month',
      description: 'Perfect for trying premium features',
      popular: false,
      icon: Zap
    },
    '3_months': {
      duration: 3,
      price: 4000,
      originalPrice: 4500,
      discount: 11,
      label: '3 Months',
      description: 'Great value for regular users',
      popular: true,
      icon: Sparkles
    },
    '6_months': {
      duration: 6,
      price: 7500,
      originalPrice: 9000,
      discount: 17,
      label: '6 Months',
      description: 'Best value for power users',
      popular: false,
      icon: Crown
    }
  };

  const currentPlan = subscriptionPlans[selectedPlan];

  // Check subscription status on component mount
  useEffect(() => {
    checkSubscriptionStatus();
  }, []);

  // Load Paystack script and get email
  useEffect(() => {
    const userData = localStorage.getItem('user_data');
    if (userData) {
      try {
        const parsedData = JSON.parse(userData);
        if (parsedData.email) {
          setEmail(parsedData.email);
        }
      } catch (err) {
        console.error('Error parsing user_data from localStorage:', err);
      }
    }

    // Load Paystack script
    if (window.PaystackPop) {
      setPaystackLoaded(true);
    } else {
      const script = document.createElement('script');
      script.src = 'https://js.paystack.co/v1/inline.js';
      script.async = true;
      script.onload = () => setPaystackLoaded(true);
      script.onerror = () =>
        setError('Failed to load payment system. Please check your internet connection.');
      document.body.appendChild(script);
    }
  }, []);

  const checkSubscriptionStatus = async () => {
    setCheckingStatus(true);
    try {
      const token = localStorage.getItem('auth_token') || '';
      const apiUrl = import.meta.env.VITE_API_URL || 'http://localhost:8000';
      
      const response = await fetch(`${apiUrl}/api/subscription-status`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
        },
      });

      if (response.ok) {
        const data = await response.json();
        setSubscriptionStatus(data);
      }
    } catch (err) {
      console.error('Error checking subscription status:', err);
    } finally {
      setCheckingStatus(false);
    }
  };

  const getMonthlySavings = (plan) => {
    const monthlyPrice = subscriptionPlans['1_month'].price;
    const totalMonthlyPrice = monthlyPrice * plan.duration;
    return totalMonthlyPrice - plan.price;
  };

  const handlePayment = () => {
    setLoading(true);
    setError('');

    if (!paystackLoaded || !window.PaystackPop) {
      setError('Payment system is still loading. Please wait a moment and try again.');
      setLoading(false);
      return;
    }

    if (!email.trim()) {
      setError('Email address is missing. Please ensure you are logged in.');
      setLoading(false);
      return;
    }

    const paystackKey = import.meta.env.VITE_PAYSTACK_PUBLIC_KEY || 'pk_test_demo';
    
    try {
      const handler = window.PaystackPop?.setup({
        key: paystackKey,
        email: email.trim(),
        amount: currentPlan.price * 100,
        currency: 'NGN',
        ref: `premium_${selectedPlan}_${Date.now()}`,
        callback: (response) => {
          console.log('Payment Successful:', response);
          setLoading(false);
          verifyTransaction(response.reference);
        },
        onClose: () => setLoading(false),
      });

      handler?.openIframe();
    } catch (err) {
      setError('Failed to initialize payment. Please try again.');
      setLoading(false);
    }
  };

  const verifyTransaction = async (reference) => {
    setSaving(true);
    setError('');
  
    try {
      const token = localStorage.getItem('auth_token') || '';
      const apiUrl = import.meta.env.VITE_API_URL || 'http://localhost:8000';
      
      const response = await fetch(`${apiUrl}/api/verify-payment`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`,
        },
        body: JSON.stringify({
          reference: reference,
          email: email.trim(),
          plan: selectedPlan,
        })
      });

      const data = await response.json();

      if (response.status === 422) {
        setError(data.message || 'Validation error. Please try again.');
        return;
      }

      if (!response.ok) {
        setError(data.message || 'Payment verification failed. Please contact support with reference: ' + reference);
        return;
      }

      if (data.message && data.subscription) {
        setSuccessData({
          status: 'success',
          plan: currentPlan,
          reference: reference,
          email: email.trim(),
          subscription: data.subscription,
          message: data.message
        });
        setShowSuccess(true);
      } else {
        setError('Payment verification failed. Please contact support with reference: ' + reference);
      }
    } catch (err) {
      console.error('Verification error:', err);
      setError('Unable to verify payment. Please check your connection and try again. Reference: ' + reference);
    } finally {
      setSaving(false);
    }
  };

  const handleSuccessClose = () => {
    setShowSuccess(false);
    // Refresh subscription status after successful payment
    checkSubscriptionStatus();
  };
  
  const isProcessing = loading || saving;

  // Format date helper
  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', { 
      year: 'numeric', 
      month: 'long', 
      day: 'numeric' 
    });
  };

  // Success Popup Component
  const SuccessPopup = () => (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-2xl shadow-xl max-w-md w-full mx-4 overflow-hidden">
        <div className="bg-indigo-600 text-white p-6 text-center">
          <div className="w-16 h-16 bg-white bg-opacity-20 rounded-full mx-auto mb-4 flex items-center justify-center">
            <Check className="w-8 h-8 text-white" />
          </div>
          <h3 className="text-xl font-bold mb-2">Payment Successful!</h3>
          <p className="text-indigo-100 text-sm">Your premium subscription is now active</p>
        </div>

        <div className="p-6">
          <div className="space-y-4 mb-6">
            <div className="flex justify-between items-center py-2 border-b border-gray-100">
              <span className="text-gray-600">Plan</span>
              <span className="font-semibold text-gray-900">{successData?.plan?.label}</span>
            </div>
            <div className="flex justify-between items-center py-2 border-b border-gray-100">
              <span className="text-gray-600">Amount</span>
              <span className="font-semibold text-gray-900">₦{successData?.plan?.price?.toLocaleString()}</span>
            </div>
            <div className="flex justify-between items-center py-2 border-b border-gray-100">
              <span className="text-gray-600">Email</span>
              <span className="font-semibold text-sm text-gray-900">{successData?.email}</span>
            </div>
            <div className="flex justify-between items-center py-2">
              <span className="text-gray-600">Reference</span>
              <span className="font-mono text-xs text-gray-500">{successData?.reference?.slice(-8)}</span>
            </div>
          </div>

          <div className="bg-indigo-50 border border-indigo-200 rounded-lg p-4 mb-6">
            <p className="text-indigo-900 text-sm text-center font-medium">
              🎉 Welcome to Premium! You now have access to all premium features.
            </p>
          </div>

          <button
            onClick={handleSuccessClose}
            className="w-full bg-indigo-600 text-white py-3 rounded-lg hover:bg-indigo-700 transition-all duration-200 font-semibold shadow-sm"
          >
            Continue
          </button>
        </div>
      </div>
    </div>
  );

  // Already Premium Display
  const AlreadyPremium = () => (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-12">
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8 text-center">
          <div className="w-20 h-20 bg-indigo-100 rounded-full mx-auto mb-6 flex items-center justify-center">
            <Crown className="w-10 h-10 text-indigo-600" />
          </div>
          
          <h2 className="text-3xl font-bold text-gray-900 mb-4">
            You're Already Premium!
          </h2>
          
          <p className="text-gray-600 mb-8 text-lg">
            You have an active premium subscription
          </p>

          {subscriptionStatus && (
            <div className="bg-indigo-50 border border-indigo-200 rounded-lg p-6 mb-8 text-left max-w-md mx-auto">
              <div className="space-y-3">
                <div className="flex justify-between items-center">
                  <span className="text-gray-700 font-medium">Plan</span>
                  <span className="text-gray-900 font-semibold">
                    {subscriptionPlans[subscriptionStatus.plan]?.label || subscriptionStatus.plan}
                  </span>
                </div>
                <div className="flex justify-between items-center">
                  <span className="text-gray-700 font-medium">Status</span>
                  <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                    {subscriptionStatus.status}
                  </span>
                </div>
                {subscriptionStatus.starts_at && (
                  <div className="flex justify-between items-center">
                    <span className="text-gray-700 font-medium">Started</span>
                    <span className="text-gray-900">{formatDate(subscriptionStatus.starts_at)}</span>
                  </div>
                )}
                {subscriptionStatus.ends_at && (
                  <div className="flex justify-between items-center">
                    <span className="text-gray-700 font-medium">Expires</span>
                    <span className="text-gray-900">{formatDate(subscriptionStatus.ends_at)}</span>
                  </div>
                )}
              </div>
            </div>
          )}

          <div className="space-y-4">
            <button
              onClick={() => navigate('/dashboard')}
              className="w-full max-w-md mx-auto block bg-indigo-600 text-white py-3 px-6 rounded-lg hover:bg-indigo-700 transition-all duration-200 font-semibold"
            >
              Go to Dashboard
            </button>
            
            <p className="text-gray-500 text-sm">
              Want to extend your subscription? Contact support.
            </p>
          </div>
        </div>
      </div>
    </div>
  );

  // Loading state
  if (checkingStatus) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="w-16 h-16 border-4 border-indigo-600 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
          <p className="text-gray-600">Checking subscription status...</p>
        </div>
      </div>
    );
  }

  // Show "Already Premium" if user has active subscription
  if (subscriptionStatus && subscriptionStatus.status === 'active') {
    return <AlreadyPremium />;
  }

  // Show payment page
  return (
    <div className="min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 py-12">
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold text-gray-900 mb-4">
            Upgrade to Premium
          </h1>
          <p className="text-xl text-gray-600 max-w-2xl mx-auto">
            Unlock all features and get the most out of your experience
          </p>
        </div>

        {/* Plan Selection */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-12">
          {Object.entries(subscriptionPlans).map(([planKey, plan]) => {
            const IconComponent = plan.icon;
            const savings = getMonthlySavings(plan);
            
            return (
              <div
                key={planKey}
                className={`relative bg-white rounded-xl border-2 transition-all duration-200 cursor-pointer hover:shadow-md ${
                  selectedPlan === planKey 
                    ? 'border-indigo-600 shadow-md' 
                    : 'border-gray-200 hover:border-gray-300'
                } ${plan.popular ? 'ring-2 ring-indigo-500 ring-opacity-20' : ''}`}
                onClick={() => setSelectedPlan(planKey)}
              >
                {plan.popular && (
                  <div className="absolute -top-3 left-1/2 transform -translate-x-1/2">
                    <span className="bg-indigo-600 text-white px-3 py-1 rounded-full text-xs font-semibold">
                      Most Popular
                    </span>
                  </div>
                )}

                <div className="p-6">
                  <div className="flex items-center justify-between mb-4">
                    <IconComponent className="w-8 h-8 text-indigo-600" />
                    {selectedPlan === planKey && (
                      <div className="w-6 h-6 bg-indigo-600 rounded-full flex items-center justify-center">
                        <Check className="w-4 h-4 text-white" />
                      </div>
                    )}
                  </div>

                  <h3 className="text-xl font-bold text-gray-900 mb-2">{plan.label}</h3>
                  <p className="text-gray-600 text-sm mb-4">{plan.description}</p>

                  <div className="mb-4">
                    <div className="flex items-baseline">
                      <span className="text-3xl font-bold text-gray-900">₦{plan.price.toLocaleString()}</span>
                      {plan.discount > 0 && (
                        <span className="ml-2 text-lg text-gray-500 line-through">₦{plan.originalPrice.toLocaleString()}</span>
                      )}
                    </div>
                    {plan.discount > 0 && (
                      <p className="text-indigo-600 text-sm font-semibold mt-1">
                        Save ₦{savings.toLocaleString()} ({plan.discount}% off)
                      </p>
                    )}
                  </div>

                  <div className="text-gray-500 text-sm font-medium">
                    ₦{Math.round(plan.price / plan.duration).toLocaleString()} per month
                  </div>
                </div>
              </div>
            );
          })}
        </div>

        {/* Payment Section */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-8">
          <h3 className="text-xl font-semibold text-gray-900 mb-6">
            Complete Your Purchase
          </h3>
          
          {/* Plan Summary */}
          <div className="bg-gray-50 border border-gray-200 rounded-lg p-4 mb-6">
            <div className="flex justify-between items-center">
              <div>
                <p className="font-semibold text-gray-900">{currentPlan.label} Plan</p>
                <p className="text-gray-600 text-sm">{currentPlan.description}</p>
              </div>
              <div className="text-right">
                <p className="text-2xl font-bold text-gray-900">₦{currentPlan.price.toLocaleString()}</p>
                {currentPlan.discount > 0 && (
                  <p className="text-indigo-600 text-sm font-semibold">Save {currentPlan.discount}%</p>
                )}
              </div>
            </div>
          </div>

          {/* Error Display */}
          {error && (
            <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-6 flex items-center">
              <X className="w-5 h-5 text-red-500 mr-2 flex-shrink-0" />
              <p className="text-red-800 text-sm">{error}</p>
            </div>
          )}

          {/* Payment Form */}
          <div className="space-y-6">
            <div>
              <label htmlFor="email" className="block text-sm font-semibold text-gray-900 mb-2">
                Email Address
              </label>
              <input
                type="email"
                id="email"
                value={email}
                readOnly
                className="w-full px-4 py-3 rounded-lg border border-gray-300 bg-gray-100 cursor-not-allowed text-gray-700"
              />
            </div>

            <button
              onClick={handlePayment}
              className={`w-full py-4 px-6 rounded-lg text-white font-semibold transition-all duration-200 flex items-center justify-center shadow-sm ${
                isProcessing || !paystackLoaded || !email.trim()
                  ? 'bg-gray-400 cursor-not-allowed' 
                  : 'bg-indigo-600 hover:bg-indigo-700 active:scale-95'
              }`}
              disabled={isProcessing || !paystackLoaded || !email.trim()}
            >
              {saving ? (
                <>
                  <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div>
                  Verifying Payment...
                </>
              ) : loading ? (
                <>
                  <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin mr-2"></div>
                  Processing...
                </>
              ) : (
                `Pay ₦${currentPlan.price.toLocaleString()} - ${currentPlan.label}`
              )}
            </button>

            <p className="text-center text-gray-500 text-sm">
              Secure payment powered by Paystack
            </p>
          </div>
        </div>
      </div>

      {/* Success Popup */}
      {showSuccess && <SuccessPopup />}
    </div>
  );
}

export default Paystack;