import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { User, BookOpen, Crown, Brain, Edit2, Upload, Save, X, CheckCircle, AlertCircle, FileText, Calendar, Award, Activity, Settings, MapPin, Mail, GraduationCap, Building, Trash2, Camera } from 'lucide-react';
import { useStats } from './Dashboard'; // Adjust path as needed

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL; // Replace with your actual API base URL

const Profile = () => {
  const { userProfile, stats, recentActivity, achievements, loading } = useStats();
  const navigate = useNavigate();

  const [isEditing, setIsEditing] = useState(false);
  const [editedProfile, setEditedProfile] = useState({ ...userProfile });
  const [activeTab, setActiveTab] = useState('overview');
  const [toast, setToast] = useState({ show: false, message: '', type: 'info' });
  const [settings, setSettings] = useState({
    emailNotifications: true,
    weeklySummary: true,
    achievementNotifications: false,
    showProfile: true,
    showActivity: true,
  });

  // Sync editedProfile when userProfile changes
  useEffect(() => {
    setEditedProfile({ ...userProfile });
  }, [userProfile]);

  // Toast notification handlers
  const showToast = (message, type) => {
    setToast({ show: true, message, type });
    setTimeout(() => setToast({ show: false, message: '', type: 'info' }), 4000);
  };

  const hideToast = () => {
    setToast({ show: false, message: '', type: '' });
  };

  // Toast notification component
  const Toast = ({ show, message, type, onClose }) => {
    useEffect(() => {
      if (show) {
        const timer = setTimeout(() => {
          onClose();
        }, 4000);
        return () => clearTimeout(timer);
      }
    }, [show, onClose]);

    if (!show) return null;

    return (
      <div
        className={`fixed top-4 right-4 z-50 p-4 rounded-xl shadow-2xl backdrop-blur-sm transform transition-all duration-300 ease-in-out ${
          show ? 'translate-y-0 opacity-100' : '-translate-y-2 opacity-0'
        } ${
          type === 'success'
            ? 'bg-gradient-to-r from-green-500 to-emerald-500 text-white'
            : type === 'error'
            ? 'bg-gradient-to-r from-red-500 to-pink-500 text-white'
            : 'bg-gradient-to-r from-blue-500 to-indigo-500 text-white'
        }`}
      >
        <div className="flex items-center space-x-3">
          {type === 'success' ? (
            <CheckCircle className="h-5 w-5 flex-shrink-0" />
          ) : (
            <AlertCircle className="h-5 w-5 flex-shrink-0" />
          )}
          <span className="font-medium">{message}</span>
          <button
            onClick={onClose}
            className="ml-2 text-white hover:text-gray-200 transition-colors"
          >
            <X className="h-4 w-4" />
          </button>
        </div>
      </div>
    );
  };

  // Handle input change
  const handleInputChange = (field, value) => {
    setEditedProfile((prev) => ({
      ...prev,
      [field]: value,
    }));
  };

  // Handle settings change
  const handleSettingsChange = (field) => {
    setSettings((prev) => ({ ...prev, [field]: !prev[field] }));
  };

  // Validate form
  const validateForm = () => {
    if (!editedProfile.name.trim()) {
      showToast('Name is required', 'error');
      return false;
    }
    if (!editedProfile.email.trim()) {
      showToast('Email is required', 'error');
      return false;
    }
    if (!editedProfile.year.trim()) {
      showToast('Academic year is required', 'error');
      return false;
    }
    if (!editedProfile.university.trim()) {
      showToast('University is required', 'error');
      return false;
    }
    return true;
  };

  // Handle profile save
  const handleSave = async () => {
    if (!validateForm()) return;

    setLoading((prev) => ({ ...prev, save: true }));
    try {
      const response = await fetch(`${API_BASE_URL}/dashboard/profile`, {
        method: 'PUT',
        headers: {
          Authorization: `Bearer ${localStorage.getItem('auth_token')}`,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(editedProfile),
      });

      if (!response.ok) {
        throw new Error(`HTTP error! Status: ${response.status}`);
      }

      // Update userProfile in parent component if needed
      setIsEditing(false);
      showToast('Profile updated successfully!', 'success');
    } catch (error) {
      console.error('Error updating profile:', error);
      showToast('Failed to update profile. Please try again.', 'error');
    } finally {
      setLoading((prev) => ({ ...prev, save: false }));
    }
  };

  // Handle cancel edit
  const handleCancel = () => {
    setEditedProfile({ ...userProfile });
    setIsEditing(false);
  };

  // Handle account deletion
  const handleDeleteAccount = async () => {
    if (!window.confirm('Are you sure you want to delete your account? This action cannot be undone.')) return;

    setLoading((prev) => ({ ...prev, save: true }));
    try {
      const response = await fetch(`${API_BASE_URL}/dashboard/profile`, {
        method: 'DELETE',
        headers: {
          Authorization: `Bearer ${localStorage.getItem('auth_token')}`,
          'Content-Type': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! Status: ${response.status}`);
      }

      showToast('Account deleted successfully. You will be logged out.', 'success');
      setTimeout(() => {
        localStorage.removeItem('auth_token'); // Fixed 'token' to 'auth_token' for consistency
        navigate('/register');
      }, 2000);
    } catch (error) {
      console.error('Error deleting account:', error);
      showToast('Failed to delete account. Please try again.', 'error');
    } finally {
      setLoading((prev) => ({ ...prev, save: false }));
    }
  };

  // Icon mapping functions
  const getActivityIcon = (iconName) => {
    const icons = {
      Upload: <Upload className="h-4 w-4" />,
      CheckCircle: <CheckCircle className="h-4 w-4" />,
      Calendar: <Calendar className="h-4 w-4" />,
      Award: <Award className="h-4 w-4" />,
    };
    return icons[iconName] || <User className="h-4 w-4" />;
  };

  const getAchievementIcon = (iconName) => {
    const icons = {
      BookOpen: <BookOpen className="h-5 w-5" />,
      Calendar: <Calendar className="h-5 w-5" />,
      Crown: <Crown className="h-5 w-5" />,
      Brain: <Brain className="h-5 w-5" />,
    };
    return icons[iconName] || <Award className="h-5 w-5" />;
  };

  return (
    <div className="min-h-screen p-4 sm:p-6 lg:p-8">
      <Toast show={toast.show} message={toast.message} type={toast.type} onClose={hideToast} />

      <div className="max-w-4xl mx-auto">
        {/* Header */}
        <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
          <div>
            <h1 className="text-3xl font-bold text-gray-800">Profile</h1>
            <p className="text-sm text-gray-600 mt-1">Manage your account and track your progress</p>
          </div>
          {/* <div className="flex items-center space-x-3 mt-4 md:mt-0">
            {!isEditing ? (
              <button
                onClick={() => setIsEditing(true)}
                className="flex items-center px-4 py-2 bg-indigo-600 text-white font-medium rounded-lg hover:bg-indigo-700 focus:ring-4 focus:ring-indigo-200 transition-all duration-200"
                disabled={loading.profile || loading.stats}
              >
                <Edit2 className="h-4 w-4 mr-2" />
                Edit Profile
              </button>
            ) : (
              <div className="flex items-center space-x-2">
                <button
                  onClick={handleCancel}
                  className="flex items-center px-4 py-2 bg-gray-500 text-white font-medium rounded-lg hover:bg-gray-600 focus:ring-4 focus:ring-gray-200 transition-all duration-200"
                  disabled={loading.save}
                >
                  <X className="h-4 w-4 mr-2" />
                  Cancel
                </button>
                <button
                  onClick={handleSave}
                  disabled={loading.save}
                  className="flex items-center px-4 py-2 bg-indigo-600 text-white font-medium rounded-lg hover:bg-indigo-700 focus:ring-4 focus:ring-indigo-200 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {loading.save ? (
                    <>
                      <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                      Saving...
                    </>
                  ) : (
                    <>
                      <Save className="h-4 w-4 mr-2" />
                      Save Changes
                    </>
                  )}
                </button>
              </div>
            )}
          </div> */}
        </div>

        {/* Profile Card */}
        <div className="bg-white rounded-xl shadow-lg border border-gray-100 overflow-hidden">
          {/* Profile Header */}
          <div className="bg-gradient-to-r from-indigo-500 to-purple-600 px-6 py-8">
            <div className="flex flex-col md:flex-row items-center md:items-start space-y-4 md:space-y-0 md:space-x-6">
              <div className="relative">
                <div className="w-20 h-20 bg-white/20 backdrop-blur-sm rounded-full flex items-center justify-center border-4 border-white/30">
                  <User className="text-white" size={36} />
                </div>
                {isEditing && (
                  <button className="absolute -bottom-1 -right-1 w-8 h-8 bg-white text-indigo-600 rounded-full flex items-center justify-center hover:bg-gray-100 transition-colors">
                    <Camera className="h-4 w-4" />
                  </button>
                )}
              </div>
              <div className="flex-1 text-center md:text-left">
                <h2 className="text-2xl font-bold text-white">{userProfile.name || 'User'}</h2>
                <p className="text-indigo-100">{userProfile.year || 'N/A'} • {userProfile.university || 'N/A'}</p>
                <div className="flex items-center justify-center md:justify-start mt-2">
                  <MapPin className="h-4 w-4 text-indigo-200 mr-1" />
                  <span className="text-indigo-100 text-sm">{userProfile.location || 'N/A'}</span>
                </div>
              </div>
              <div className="flex items-center space-x-6">
                <div className="text-center">
                  <div className="text-2xl font-bold text-white">{stats.totalPoints}</div>
                  <div className="text-indigo-200 text-sm">Points</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-white">{stats.studyStreak}</div>
                  <div className="text-indigo-200 text-sm">Day Streak</div>
                </div>
              </div>
            </div>
          </div>

          {/* Tabs */}
          <div className="border-b border-gray-200">
            <nav className="flex px-6">
              {[
                { id: 'overview', label: 'Overview', icon: <User className="h-4 w-4" /> },
                { id: 'activity', label: 'Activity', icon: <Activity className="h-4 w-4" /> },
                { id: 'achievements', label: 'Achievements', icon: <Award className="h-4 w-4" /> },
                // { id: 'settings', label: 'Settings', icon: <Settings className="h-4 w-4" /> },
              ].map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`flex items-center px-4 py-3 text-sm font-medium border-b-2 transition-all duration-200 ${
                    activeTab === tab.id
                      ? 'border-indigo-500 text-indigo-600'
                      : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }`}
                >
                  {tab.icon}
                  <span className="ml-2">{tab.label}</span>
                </button>
              ))}
            </nav>
          </div>

          {/* Tab Content */}
          <div className="p-6">
            {loading.profile || loading.stats ? (
              <div className="text-center py-12">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600 mx-auto mb-4"></div>
                <p className="text-gray-500 text-lg">Loading profile...</p>
              </div>
            ) : (
              <>
                {activeTab === 'overview' && (
                  <div className="space-y-6">
                    {/* Stats Cards */}
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                      {/* <div className="bg-blue-50 p-4 rounded-lg text-center">
                        <FileText className="h-8 w-8 text-blue-600 mx-auto mb-2" />
                        <div className="text-2xl font-bold text-blue-600">{stats.questionsUploaded}</div>
                        <div className="text-sm text-blue-600">Questions Uploaded</div>
                      </div> */}
                      <div className="bg-green-50 p-4 rounded-lg text-center">
                        <CheckCircle className="h-8 w-8 text-green-600 mx-auto mb-2" />
                        <div className="text-2xl font-bold text-green-600">{stats.questionsAnswered}</div>
                        <div className="text-sm text-green-600">Questions Answered</div>
                      </div>
                      <div className="bg-purple-50 p-4 rounded-lg text-center">
                        <Calendar className="h-8 w-8 text-purple-600 mx-auto mb-2" />
                        <div className="text-2xl font-bold text-purple-600">{stats.studyStreak}</div>
                        <div className="text-sm text-purple-600">Day Streak</div>
                      </div>
                      <div className="bg-yellow-50 p-4 rounded-lg text-center">
                        <Award className="h-8 w-8 text-yellow-600 mx-auto mb-2" />
                        <div className="text-2xl font-bold text-yellow-600">{stats.totalPoints}</div>
                        <div className="text-sm text-yellow-600">Total Points</div>
                      </div>
                    </div>

                    {/* Profile Form */}
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                      <div className="space-y-2">
                        <label className="block text-sm font-medium text-gray-700">Full Name</label>
                        <div className="relative">
                          <User className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                          <input
                            type="text"
                            value={editedProfile.name}
                            onChange={(e) => handleInputChange('name', e.target.value)}
                            className={`w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-200 ${
                              !isEditing ? 'bg-gray-50' : 'bg-white'
                            }`}
                            readOnly={!isEditing}
                            placeholder="Enter your name"
                          />
                        </div>
                      </div>
                      <div className="space-y-2">
                        <label className="block text-sm font-medium text-gray-700">Email Address</label>
                        <div className="relative">
                          <Mail className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                          <input
                            type="email"
                            value={editedProfile.email}
                            onChange={(e) => handleInputChange('email', e.target.value)}
                            className={`w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-200 ${
                              !isEditing ? 'bg-gray-50' : 'bg-white'
                            }`}
                            readOnly
                          />
                        </div>
                      </div>
                      <div className="space-y-2">
                        <label className="block text-sm font-medium text-gray-700">Academic Year</label>
                        <div className="relative">
                          <GraduationCap className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                          <input
                            type="text"
                            value={editedProfile.year}
                            onChange={(e) => handleInputChange('year', e.target.value)}
                            className={`w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-200 ${
                              !isEditing ? 'bg-gray-50' : 'bg-white'
                            }`}
                            readOnly={!isEditing}
                            placeholder="Enter your academic year"
                          />
                        </div>
                      </div>
                      <div className="space-y-2">
                        <label className="block text-sm font-medium text-gray-700">University</label>
                        <div className="relative">
                          <Building className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                          <input
                            type="text"
                            value={editedProfile.university}
                            onChange={(e) => handleInputChange('university', e.target.value)}
                            className={`w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-200 ${
                              !isEditing ? 'bg-gray-50' : 'bg-white'
                            }`}
                            readOnly={!isEditing}
                            placeholder="Enter your university"
                          />
                        </div>
                      </div>
                      <div className="space-y-2 md:col-span-2">
                        <label className="block text-sm font-medium text-gray-700">Bio</label>
                        <div className="relative">
                          <textarea
                            value={editedProfile.bio}
                            onChange={(e) => handleInputChange('bio', e.target.value)}
                            rows={3}
                            className={`w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-200 resize-none ${
                              !isEditing ? 'bg-gray-50' : 'bg-white'
                            }`}
                            readOnly={!isEditing}
                            placeholder="Tell us a bit about yourself..."
                          />
                        </div>
                      </div>
                      <div className="space-y-2 md:col-span-2">
                        <label className="block text-sm font-medium text-gray-700">Location</label>
                        <div className="relative">
                          <MapPin className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                          <input
                            type="text"
                            value={editedProfile.location}
                            onChange={(e) => handleInputChange('location', e.target.value)}
                            className={`w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-200 ${
                              !isEditing ? 'bg-gray-50' : 'bg-white'
                            }`}
                            readOnly={!isEditing}
                            placeholder="Enter your location"
                          />
                        </div>
                      </div>
                    </div>
                  </div>
                )}

                {activeTab === 'activity' && (
                  <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-800 mb-4">Recent Activity</h3>
                    {recentActivity.length === 0 ? (
                      <div className="text-center py-6 text-gray-500">No recent activity found</div>
                    ) : (
                      recentActivity.map((performance, index) => (
                        <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                          <div>
                            <h4 className="font-semibold text-gray-800">{performance.subject}</h4>
                            <p className="text-sm text-gray-600">
                              {performance.total_questions} questions • {performance.time}
                            </p>
                          </div>
                          <div
                            className={`px-3 py-1 rounded-full text-sm font-bold ${
                              performance.score >= 80
                                ? 'bg-green-100 text-green-800'
                                : performance.score >= 70
                                ? 'bg-yellow-100 text-yellow-800'
                                : 'bg-red-100 text-red-800'
                            }`}
                          >
                            {performance.score}%
                          </div>
                        </div>
                      ))
                    )}
                  </div>
                )}

                {activeTab === 'achievements' && (
                  <div className="space-y-4">
                    <h3 className="text-lg font-semibold text-gray-800 mb-4">Achievements</h3>
                    {achievements.length === 0 ? (
                      <div className="text-center py-8 text-gray-500">No achievements unlocked yet</div>
                    ) : (
                      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                        {achievements.map((achievement) => (
                          <div
                            key={achievement.id}
                            className={`p-4 rounded-lg border-2 transition-all duration-200 ${
                              achievement.earned
                                ? 'border-indigo-200 bg-indigo-50'
                                : 'border-gray-200 bg-gray-50 opacity-60'
                            }`}
                          >
                            <div className="flex items-start space-x-3">
                              <div
                                className={`flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center ${
                                  achievement.earned ? 'bg-indigo-600 text-white' : 'bg-gray-400 text-white'
                                }`}
                              >
                                {getAchievementIcon(achievement.icon)}
                              </div>
                              <div className="flex-1">
                                <h4
                                  className={`font-semibold ${
                                    achievement.earned ? 'text-indigo-800' : 'text-gray-600'
                                  }`}
                                >
                                  {achievement.title}
                                </h4>
                                <p
                                  className={`text-sm ${
                                    achievement.earned ? 'text-indigo-600' : 'text-gray-500'
                                  }`}
                                >
                                  {achievement.description}
                                </p>
                                {achievement.earned ? (
                                  <p className="text-sm text-gray-500 mt-1">
                                    Earned on {new Date(achievement.earned_at).toLocaleDateString()}
                                  </p>
                                ) : achievement.progress && achievement.target ? (
                                  <div className="mt-2">
                                    <div className="text-sm text-gray-500">
                                      Progress: {achievement.progress}/{achievement.target}
                                    </div>
                                    <div className="w-full bg-gray-200 rounded-full h-2.5 mt-1">
                                      <div
                                        className="bg-indigo-600 h-2.5 rounded-full"
                                        style={{
                                          width: `${(achievement.progress / achievement.target) * 100}%`,
                                        }}
                                      ></div>
                                    </div>
                                  </div>
                                ) : null}
                              </div>
                              {achievement.earned && (
                                <svg
                                  className="h-5 w-5 text-green-500"
                                  fill="none"
                                  stroke="currentColor"
                                  viewBox="0 0 24 24"
                                  xmlns="http://www.w3.org/2000/svg"
                                >
                                  <path
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                    strokeWidth="2"
                                    d="M5 13l4 4L19 7"
                                  ></path>
                                </svg>
                              )}
                            </div>
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                )}

                {activeTab === 'settings' && (
                  <div className="space-y-6">
                    <h3 className="text-lg font-semibold text-gray-800 mb-4">Account Settings</h3>
                    <div className="bg-gray-50 p-4 rounded-lg">
                      <h4 className="font-medium text-gray-800 mb-3">Notifications</h4>
                      <div className="space-y-3">
                        <label className="flex items-center">
                          <input
                            type="checkbox"
                            checked={settings.emailNotifications}
                            onChange={() => handleSettingsChange('emailNotifications')}
                            className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                          />
                          <span className="ml-2 text-sm text-gray-700">Email notifications for new questions</span>
                        </label>
                        <label className="flex items-center">
                          <input
                            type="checkbox"
                            checked={settings.weeklySummary}
                            onChange={() => handleSettingsChange('weeklySummary')}
                            className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                          />
                          <span className="ml-2 text-sm text-gray-700">Weekly progress summary</span>
                        </label>
                        <label className="flex items-center">
                          <input
                            type="checkbox"
                            checked={settings.achievementNotifications}
                            onChange={() => handleSettingsChange('achievementNotifications')}
                            className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                          />
                          <span className="ml-2 text-sm text-gray-700">Achievement notifications</span>
                        </label>
                      </div>
                    </div>
                    <div className="bg-gray-50 p-4 rounded-lg">
                      <h4 className="font-medium text-gray-800 mb-3">Privacy</h4>
                      <div className="space-y-3">
                        <label className="flex items-center">
                          <input
                            type="checkbox"
                            checked={settings.showProfile}
                            onChange={() => handleSettingsChange('showProfile')}
                            className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                          />
                          <span className="ml-2 text-sm text-gray-700">Show my profile to other users</span>
                        </label>
                        <label className="flex items-center">
                          <input
                            type="checkbox"
                            checked={settings.showActivity}
                            onChange={() => handleSettingsChange('showActivity')}
                            className="rounded border-gray-300 text-indigo-600 focus:ring-indigo-500"
                          />
                          <span className="ml-2 text-sm text-gray-700">Allow others to see my activity</span>
                        </label>
                      </div>
                    </div>
                    <div className="bg-red-50 p-4 rounded-lg border border-red-200">
                      <h4 className="font-medium text-red-800 mb-3">Danger Zone</h4>
                      <div className="space-y-3">
                        <button
                          onClick={handleDeleteAccount}
                          disabled={loading.save}
                          className="flex items-center px-4 py-2 bg-red-600 text-white font-medium rounded-lg hover:bg-red-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                        >
                          <Trash2 className="h-4 w-4 mr-2" />
                          Delete Account
                        </button>
                        <p className="text-xs text-red-600">This action cannot be undone</p>
                      </div>
                    </div>
                  </div>
                )}
              </>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default Profile;