import React, { useState, useRef, useEffect } from 'react';
import { Upload, FileText, FileImage, AlertCircle, CheckCircle, Loader, X, Eye } from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import RequirePremium from './RequirePremium';

// Success Modal - Clean and simple (no fetching logic here)
const SuccessModal = ({ isOpen, onClose, questionCount, courseName, onViewQuestions }) => {
  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-3xl shadow-2xl max-w-md w-full p-8 transform transition-all duration-300">
        <div className="text-center">
          <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <CheckCircle className="w-10 h-10 text-green-500" aria-hidden="true" />
          </div>
          
          <h3 className="text-2xl font-bold text-gray-800 mb-2">
            Questions Generated Successfully!
          </h3>
          
          <p className="text-gray-600 mb-6">
            We've generated <span className="font-semibold text-indigo-600">{questionCount} questions</span> for your course "<span className="font-semibold text-indigo-600">{courseName}</span>". 
            They have been saved to your library.
          </p>
          
          <div className="space-y-3">
            <button
              onClick={onViewQuestions}
              className="w-full bg-gradient-to-r from-indigo-600 to-purple-600 text-white py-3 px-6 rounded-xl font-semibold hover:from-indigo-700 hover:to-purple-700 transition-all duration-300 flex items-center justify-center space-x-2"
              aria-label="View generated questions"
            >
              <Eye className="w-5 h-5" aria-hidden="true" />
              <span>View Questions</span>
            </button>
            
            <button
              onClick={onClose}
              className="w-full bg-gray-100 text-gray-700 py-3 px-6 rounded-xl font-semibold hover:bg-gray-200 transition-all duration-300"
              aria-label="Generate more questions"
            >
              Generate More Questions
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};

const QuestionGenerator = ({ onNavigateToQuestions }) => {
  const navigate = useNavigate();
  const [file, setFile] = useState(null);
  const [questionCount, setQuestionCount] = useState(10);
  const [courseName, setCourseName] = useState('');
  const [loading, setLoading] = useState({
    main: false,
    subjects: false,
    subtopics: false
  });
  const [uploadProgress, setUploadProgress] = useState(0);
  const [error, setError] = useState('');
  const [isDragOver, setIsDragOver] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [generatedData, setGeneratedData] = useState({ questionCount: 0, courseName: '' });

  // Subject & Subtopic states
  const [subjects, setSubjects] = useState([]);
  const [subtopics, setSubtopics] = useState([]);
  const [quizSettings, setQuizSettings] = useState({
    subject: 'all',
    subtopic: 'all'
  });

  const fileInputRef = useRef(null);
  const API_BASE_URL = import.meta.env.VITE_API_BASE_URL;

  const getAuthHeaders = () => ({
    'Authorization': `Bearer ${localStorage.getItem('auth_token')}`,
    'Accept': 'application/json',
  });

  // Fetch subjects on component mount
  useEffect(() => {
    const fetchSubjects = async () => {
      setLoading(prev => ({ ...prev, subjects: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects`, {
          headers: getAuthHeaders(),
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubjects(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subjects:', err);
        setError('Failed to load subjects');
      } finally {
        setLoading(prev => ({ ...prev, subjects: false }));
      }
    };
    fetchSubjects();
  }, []);

  // Fetch subtopics when subject changes
  useEffect(() => {
    const fetchSubtopics = async () => {
      if (!quizSettings.subject || quizSettings.subject === 'all') {
        setSubtopics([]);
        setQuizSettings(prev => ({ ...prev, subtopic: 'all' }));
        return;
      }

      setLoading(prev => ({ ...prev, subtopics: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects/${quizSettings.subject}/subtopics`, {
          headers: getAuthHeaders(),
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubtopics(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subtopics:', err);
        setError('Failed to load subtopics');
      } finally {
        setLoading(prev => ({ ...prev, subtopics: false }));
      }
    };
    fetchSubtopics();
  }, [quizSettings.subject]);

  const handleFileChange = (selectedFile) => {
    if (!selectedFile) {
      setError('No file selected');
      setFile(null);
      return;
    }

    const allowedTypes = [
      'application/pdf',
      'application/vnd.ms-powerpoint',
      'application/vnd.openxmlformats-officedocument.presentationml.presentation',
    ];

    if (allowedTypes.includes(selectedFile.type)) {
      if (selectedFile.size > 50 * 1024 * 1024) {
        setError('File size exceeds 50MB. Please upload a smaller file.');
        setFile(null);
      } else {
        setFile(selectedFile);
        setError('');
      }
    } else {
      setError('Please select a PDF or PowerPoint file');
      setFile(null);
    }
  };

  const handleDragOver = (e) => {
    e.preventDefault();
    setIsDragOver(true);
  };

  const handleDragLeave = (e) => {
    e.preventDefault();
    setIsDragOver(false);
  };

  const handleDrop = (e) => {
    e.preventDefault();
    setIsDragOver(false);
    const droppedFile = e.dataTransfer.files[0];
    handleFileChange(droppedFile);
  };

  const removeFile = () => {
    setFile(null);
    setError('');
  };

  const resetForm = () => {
    setFile(null);
    setCourseName('');
    setQuestionCount(10);
    setError('');
  };

  const simulateUploadProgress = () => {
    setUploadProgress(0);
    const interval = setInterval(() => {
      setUploadProgress((prev) => {
        if (prev >= 90) {
          clearInterval(interval);
          return 90;
        }
        return prev + Math.random() * 10;
      });
    }, 500);
    return interval;
  };

  const handleSubmit = async (e) => {
    e.preventDefault();

    if (!file) {
      setError('Please select a file');
      return;
    }

    if (questionCount < 1 || questionCount > 50) {
      setError('Please select between 1 and 50 questions');
      return;
    }

    if (!courseName.trim()) {
      setError('Please enter a course name');
      return;
    }

    setLoading(prev => ({ ...prev, main: true }));
    setError('');

    const progressInterval = simulateUploadProgress();

    try {
      const formData = new FormData();
      formData.append('file', file);
      formData.append('question_count', questionCount);
      formData.append('course_name', courseName);

      // Optional: send subject and subtopic if selected
      if (quizSettings.subject !== 'all') {
        formData.append('subject_id', quizSettings.subject);
      }
      if (quizSettings.subtopic !== 'all') {
        formData.append('subtopic_id', quizSettings.subtopic);
      }

      const response = await fetch(`${API_BASE_URL}/ai-generate-questions`, {
        method: 'POST',
        body: formData,
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('auth_token')}`,
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.error?.message || errorData.message || `Server error: ${response.status}`);
      }

      const result = await response.json();

      if (!result.questions || !Array.isArray(result.questions)) {
        throw new Error('Invalid response format: questions array not found');
      }

      setGeneratedData({ questionCount: result.questions.length, courseName });
      setShowSuccessModal(true);
      setUploadProgress(100);
      resetForm();
    } catch (error) {
      console.error('API Error:', error);
      setError(error.message || 'An error occurred while generating questions');
    } finally {
      clearInterval(progressInterval);
      setLoading(prev => ({ ...prev, main: false }));
      setUploadProgress(0);
    }
  };

  const handleViewQuestions = () => {
    setShowSuccessModal(false);
 
    navigate('/dashboard/courses');
  };

  const getFileIcon = (fileType) => {
    if (fileType === 'application/pdf') {
      return <FileText className="w-8 h-8 text-red-500" aria-hidden="true" />;
    }
    return <FileImage className="w-8 h-8 text-blue-500" aria-hidden="true" />;
  };

  return (
    <>
    <RequirePremium>
      <div className="min-h-screen bg-gray-50 font-sans text-gray-800">
  <div className="max-w-4xl mx-auto p-6 lg:p-12">
  <div className="text-center mb-12">
  {/* Main Heading */}
  <h1 className="text-4xl sm:text-5xl font-bold text-gray-900 mb-6 leading-tight">
    AI Question Generator
  </h1>

  {/* Subheading */}
  <p className="text-lg sm:text-xl text-gray-600 max-w-3xl mx-auto leading-relaxed mb-8">
    Upload your lecture notes or study materials in PDF format, and let advanced AI instantly generate 
    high-quality multiple-choice questions tailored to your content.
  </p>

  {/* PowerPoint Conversion Helper */}
  <div className="bg-blue-50 border border-blue-200 rounded-xl p-6 max-w-2xl mx-auto">
    <div className="flex flex-col sm:flex-row items-center justify-center gap-4 text-blue-900">
      <div className="text-left">
        <p className="font-semibold text-blue-950 mb-1">
          Have PowerPoint slides?
        </p>
        <p className="text-sm text-blue-800">
          Convert them to PDF first for best results.
        </p>
      </div>
      <a
        href="https://www.ilovepdf.com/powerpoint_to_pdf"
        target="_blank"
        rel="noopener noreferrer"
        className="inline-flex items-center gap-2 px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white font-medium rounded-lg transition-colors shadow-sm"
      >
        Convert PPT to PDF
        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14" />
        </svg>
      </a>
    </div>
  </div>
</div>

    <div className="bg-white rounded-2xl shadow-sm border border-gray-200 p-8 sm:p-12">
      <div className="space-y-8">

        {/* Subject Dropdown */}
        <div>
          <label className="block text-sm font-semibold text-gray-900 mb-2">Subject</label>
          <select
            value={quizSettings.subject}
            onChange={(e) => setQuizSettings({ subject: e.target.value, subtopic: 'all' })}
            className="w-full p-3 border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none transition-all bg-white"
            disabled={loading.subjects || loading.main}
          >
            <option value="all">All Subjects</option>
            {subjects.map((subject) => (
              <option key={subject.id || subject} value={subject.id || subject}>
                {subject.name || subject}
              </option>
            ))}
          </select>
        </div>

        {/* Subtopic Dropdown */}
        <div>
          <label className="block text-sm font-semibold text-gray-900 mb-2">Subtopic</label>
          <select
            value={quizSettings.subtopic}
            onChange={(e) => setQuizSettings(prev => ({ ...prev, subtopic: e.target.value }))}
            className="w-full p-3 border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none transition-all bg-white"
            disabled={loading.subtopics || loading.main || quizSettings.subject === 'all'}
          >
            <option value="all">All Subtopics</option>
            {subtopics.map((subtopic) => (
              <option key={subtopic.id || subtopic} value={subtopic.id || subtopic}>
                {subtopic.name || subtopic}
              </option>
            ))}
          </select>

          {/* Loading Subtopics Feedback */}
          {loading.subtopics && (
            <div className="mt-3 flex items-center space-x-2 text-indigo-600">
              <Loader className="w-4 h-4 animate-spin" />
              <span className="text-sm">Loading subtopics...</span>
            </div>
          )}

          {/* Helpful message when subject is "All" */}
          {!loading.subtopics && quizSettings.subject === 'all' && (
            <p className="mt-2 text-sm text-gray-500">
              Select a subject above to load available subtopics
            </p>
          )}

          {/* No subtopics found */}
          {!loading.subtopics && quizSettings.subject !== 'all' && subtopics.length === 0 && (
            <p className="mt-2 text-sm text-gray-500">
              No subtopics found for this subject
            </p>
          )}
        </div>

        {/* Course Name */}
        <div>
          <label htmlFor="course_name" className="block text-sm font-semibold text-gray-900 mb-2">
            Course Name
          </label>
          <input
            id="course_name"
            type="text"
            value={courseName}
            onChange={(e) => setCourseName(e.target.value)}
            className="w-full p-3 border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-100 outline-none transition-all"
            placeholder="Enter course name (e.g., Biology 101, Marketing Fundamentals)"
            disabled={loading.main}
            aria-required="true"
          />
        </div>

        {/* File Upload Area */}
        <div>
          <label className="block text-sm font-semibold text-gray-900 mb-2">
            Upload Your Document
          </label>
          
          {!file ? (
            <div
              className={`relative border-2 border-dashed rounded-lg p-8 text-center transition-all duration-200 cursor-pointer ${
                isDragOver
                  ? 'border-indigo-500 bg-indigo-50'
                  : 'border-gray-300 hover:border-gray-400 hover:bg-gray-50'
              } ${loading.main ? 'pointer-events-none opacity-50' : ''}`}
              onDragOver={handleDragOver}
              onDragLeave={handleDragLeave}
              onDrop={handleDrop}
              onClick={() => !loading.main && fileInputRef.current?.click()}
              role="button"
              tabIndex={loading.main ? -1 : 0}
              onKeyDown={(e) => e.key === 'Enter' && !loading.main && fileInputRef.current?.click()}
            >
              <input
                ref={fileInputRef}
                type="file"
                accept=".pdf"
                // accept=".pdf, .pptx"
                onChange={(e) => handleFileChange(e.target.files[0])}
                className="sr-only"
                disabled={loading.main}
              />
              
              <Upload className={`w-12 h-12 mx-auto mb-4 ${isDragOver ? 'text-indigo-500' : 'text-gray-400'}`} />
              
              <h3 className="text-lg font-semibold text-gray-900 mb-2">
                {isDragOver ? 'Drop your file here' : 'Drop files here or click to browse'}
              </h3>
              <p className="text-gray-500 mb-4 text-sm">
                Supports Only PDF files up to 50MB
              </p>
              
              <div className="flex justify-center space-x-4 text-sm text-gray-500">
                <span className="flex items-center">
                  <FileText className="w-4 h-4 mr-1" /> PDF
                </span>
                {/* <span className="flex items-center">
                  <FileImage className="w-4 h-4 mr-1" /> PPT/PPTX
                </span> */}
              </div>
            </div>
          ) : (
            <div className="bg-green-50 border border-green-200 rounded-lg p-6">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  {getFileIcon(file.type)}
                  <div>
                    <h4 className="font-semibold text-gray-900">{file.name}</h4>
                    <p className="text-sm text-gray-600">
                      {(file.size / 1024 / 1024).toFixed(2)} MB
                    </p>
                  </div>
                  <CheckCircle className="w-6 h-6 text-green-600" />
                </div>
                {!loading.main && (
                  <button
                    type="button"
                    onClick={removeFile}
                    className="p-2 hover:bg-red-50 rounded-full transition-colors"
                    aria-label="Remove file"
                  >
                    <X className="w-5 h-5 text-red-600" />
                  </button>
                )}
              </div>
            </div>
          )}
        </div>

        {/* Number of Questions */}
        <div>
          <label className="block text-sm font-semibold text-gray-900 mb-3">
            Number of Questions
          </label>
          <div className="grid grid-cols-5 sm:grid-cols-10 gap-2">
            {[5, 10, 15, 20, 25, 30, 35, 40, 45, 50].map((num) => (
              <button
                key={num}
                type="button"
                disabled={loading.main}
                onClick={() => setQuestionCount(num)}
                className={`p-3 rounded-lg text-sm font-semibold transition-all ${
                  questionCount === num
                    ? 'bg-indigo-600 text-white shadow-sm'
                    : 'bg-white border border-gray-300 text-gray-700 hover:border-gray-400 hover:bg-gray-50'
                } ${loading.main ? 'opacity-50 cursor-not-allowed' : ''}`}
              >
                {num}
              </button>
            ))}
          </div>
        </div>

        {/* Error Message */}
        {error && (
          <div className="bg-red-50 border border-red-200 rounded-lg p-4 flex items-center space-x-3">
            <AlertCircle className="w-5 h-5 text-red-600 flex-shrink-0" />
            <span className="text-red-700 text-sm">{error}</span>
          </div>
        )}

        {/* Progress Bar */}
        {loading.main && (
          <div className="space-y-2">
            <div className="flex justify-between text-sm text-gray-600">
              <span>Generating questions...</span>
              <span className="font-semibold">{Math.round(uploadProgress)}%</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2 overflow-hidden">
              <div
                className="h-full bg-indigo-600 transition-all duration-300 ease-out"
                style={{ width: `${uploadProgress}%` }}
              />
            </div>
          </div>
        )}

        {/* Submit Button */}
        <button
          type="button"
          onClick={handleSubmit}
          disabled={loading.main || !file}
          className="w-full bg-indigo-600 text-white py-3.5 px-6 rounded-lg font-semibold hover:bg-indigo-700 disabled:bg-gray-300 disabled:cursor-not-allowed transition-all duration-200 flex items-center justify-center space-x-2 shadow-sm"
        >
          {loading.main ? (
            <>
              <Loader className="w-5 h-5 animate-spin" />
              <span>Generating Questions...</span>
            </>
          ) : (
            <span>Generate Questions</span>
          )}
        </button>
      </div>
    </div>
  </div>
</div>
</RequirePremium>
      <SuccessModal
        isOpen={showSuccessModal}
        onClose={() => setShowSuccessModal(false)}
        questionCount={generatedData.questionCount}
        courseName={generatedData.courseName}
        onViewQuestions={handleViewQuestions}
      />
    </>
  );
};

export default QuestionGenerator;