import React, { useState, useEffect } from 'react';
import { BookOpen, Tags, Building, BarChart, Search, X, ChevronDown, ChevronLeft, ChevronRight, FileText, HelpCircle, CheckCircle, Sparkles } from 'lucide-react';

const ViewQuestions = () => {
  const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000/api';

  // State declarations
  const [subjects, setSubjects] = useState([]);
  const [schools, setSchools] = useState([]);
  const [subtopics, setSubtopics] = useState([]);
  const [vivaTypes, setVivaTypes] = useState([]);
  const [questions, setQuestions] = useState([]);
  const [selectedSubject, setSelectedSubject] = useState('');
  const [selectedSubtopic, setSelectedSubtopic] = useState('');
  const [selectedSchool, setSelectedSchool] = useState('');
  const [selectedDifficulty, setSelectedDifficulty] = useState('all');
  const [selectedVivaType, setSelectedVivaType] = useState('all');
  const [searchQuery, setSearchQuery] = useState('');
  const [filteredQuestions, setFilteredQuestions] = useState([]);
  const [pagination, setPagination] = useState({
    current_page: 1,
    per_page: 10,
    total: 0,
    last_page: 1,
    from: 0,
    to: 0
  });
  const [loading, setLoading] = useState({
    subjects: false,
    schools: false,
    subtopics: false,
    vivaTypes: false,
    questions: false
  });
  const [toast, setToast] = useState({ show: false, message: '', type: 'info' });

  // Sample difficulties array
  const difficulties = ['Easy', 'Medium', 'Hard'];

  // Toast notification function
  const showToast = (message, type) => {
    setToast({ show: true, message, type });
    setTimeout(() => setToast({ show: false, message: '', type: 'info' }), 3000);
  };

  // Fetch subjects on component mount
  useEffect(() => {
    const fetchSubjects = async () => {
      setLoading(prev => ({ ...prev, subjects: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects`, {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
          }
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubjects(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subjects:', err);
        setSubjects([]);
        showToast('Failed to load subjects', 'error');
      } finally {
        setLoading(prev => ({ ...prev, subjects: false }));
      }
    };
    fetchSubjects();
  }, [API_BASE_URL]);

  // Fetch schools/institutions
  useEffect(() => {
    const fetchSchools = async () => {
      setLoading(prev => ({ ...prev, schools: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/institutions`, {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
          }
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSchools(data.institutions || data || []);
      } catch (err) {
        console.error('Error fetching schools:', err);
        setSchools([]);
        showToast('Failed to load institutions', 'error');
      } finally {
        setLoading(prev => ({ ...prev, schools: false }));
      }
    };
    fetchSchools();
  }, [API_BASE_URL]);

  // Fetch viva types
  useEffect(() => {
    const fetchVivaTypes = async () => {
      setLoading(prev => ({ ...prev, vivaTypes: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/viva-types`, {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
          }
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setVivaTypes(data.data || []);
      } catch (err) {
        console.error('Error fetching viva types:', err);
        setVivaTypes([]);
        showToast('Failed to load viva types', 'error');
      } finally {
        setLoading(prev => ({ ...prev, vivaTypes: false }));
      }
    };
    fetchVivaTypes();
  }, [API_BASE_URL]);

  // Fetch subtopics when subject changes
  useEffect(() => {
    const fetchSubtopics = async () => {
      if (!selectedSubject) {
        setSubtopics([]);
        setSelectedSubtopic('');
        return;
      }

      setLoading(prev => ({ ...prev, subtopics: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects/${selectedSubject}/subtopics`, {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
          }
        });
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubtopics(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subtopics:', err);
        setSubtopics([]);
        showToast('Failed to load subtopics', 'error');
      } finally {
        setLoading(prev => ({ ...prev, subtopics: false }));
      }
    };
    fetchSubtopics();
  }, [API_BASE_URL, selectedSubject]);

  // Fetch questions based on filters
  const fetchQuestions = async (page = 1) => {
    setLoading(prev => ({ ...prev, questions: true }));
    try {
      const params = new URLSearchParams();
      
      if (selectedSubject) params.append('subject', selectedSubject);
      if (selectedSubtopic) params.append('subtopic', selectedSubtopic);
      if (selectedSchool) params.append('institution', selectedSchool);
      if (selectedDifficulty && selectedDifficulty !== 'all') params.append('difficulty', selectedDifficulty);
      if (selectedVivaType && selectedVivaType !== 'all') params.append('viva_type', selectedVivaType);
      if (searchQuery.trim()) params.append('search', searchQuery.trim());
      params.append('page', page.toString());

      const queryString = params.toString();
      const url = `${API_BASE_URL}/dashboard/get-essay${queryString ? `?${queryString}` : ''}`;
      
      const response = await fetch(url, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('auth_token')}`,
          'Content-Type': 'application/json'
        }
      });
      if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
      
      const data = await response.json();
      const questionsData = data.questions || [];
      
      setQuestions(questionsData);
      setFilteredQuestions(questionsData);
      
      if (data.pagination) {
        setPagination(data.pagination);
      }
      
      if (questionsData.length === 0 && page === 1) {
        showToast('No questions found matching your criteria', 'info');
      } else if (page === 1) {
        showToast(`Found ${data.total_count || questionsData.length} question(s)`, 'success');
      }
    } catch (err) {
      console.error('Error fetching questions:', err);
      setQuestions([]);
      setFilteredQuestions([]);
      showToast('Failed to load questions', 'error');
    } finally {
      setLoading(prev => ({ ...prev, questions: false }));
    }
  };

  // Handle subject change
  const handleSubjectChange = (e) => {
    setSelectedSubject(e.target.value);
    setSelectedSubtopic('');
  };

  // Handle fetch questions button click
  const handleFetchQuestions = () => {
    fetchQuestions(1);
  };

  // Handle page change
  const handlePageChange = (page) => {
    if (page >= 1 && page <= pagination.last_page) {
      fetchQuestions(page);
    }
  };

  // Clear all filters
  const clearFilters = () => {
    setSelectedSubject('');
    setSelectedSubtopic('');
    setSelectedSchool('');
    setSelectedDifficulty('all');
    setSelectedVivaType('all');
    setSearchQuery('');
    setQuestions([]);
    setFilteredQuestions([]);
    setPagination({
      current_page: 1,
      per_page: 10,
      total: 0,
      last_page: 1,
      from: 0,
      to: 0
    });
  };

  // Generate page numbers for pagination
  const getPageNumbers = () => {
    const pages = [];
    const { current_page, last_page } = pagination;
    const maxVisible = 5;
    
    let start = Math.max(1, current_page - Math.floor(maxVisible / 2));
    let end = Math.min(last_page, start + maxVisible - 1);
    
    if (end - start + 1 < maxVisible) {
      start = Math.max(1, end - maxVisible + 1);
    }
    
    for (let i = start; i <= end; i++) {
      pages.push(i);
    }
    
    return pages;
  };

  return (
    <div className="space-y-6">
      <div className="flex flex-col md:flex-row md:items-center md:justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-800">Viva Questions & Answers</h1>
          {pagination.total > 0 && (
            <p className="text-sm text-gray-600 mt-1">
              Total: {pagination.total} questions
            </p>
          )}
        </div>
        {/* <div className="flex items-center space-x-4 mt-4 md:mt-0">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" size={20} />
            <input
              type="text"
              placeholder="Search questions..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition-all duration-200"
            />
          </div>
        </div> */}
      </div>
      
      <div className="bg-white p-6 rounded-xl shadow-lg border border-gray-100">
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 mb-6">
          <div className="space-y-3">
            <label className="block text-sm font-semibold text-gray-700">
              <BookOpen className="inline h-4 w-4 mr-2 text-indigo-600" />
              Subject
            </label>
            <div className="relative">
              <select
                value={selectedSubject}
                onChange={handleSubjectChange}
                className="w-full p-3 pr-10 text-sm text-gray-700 bg-white border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 transition-all duration-200 appearance-none"
                disabled={loading.subjects}
              >
                <option value="">
                  {loading.subjects ? 'Loading...' : 'All Subjects'}
                </option>
                {subjects.map(subject => (
                  <option key={subject.id || subject} value={subject.id || subject}>
                    {subject.name || subject}
                  </option>
                ))}
              </select>
              <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
            </div>
          </div>

          <div className="space-y-3">
            <label className="block text-sm font-semibold text-gray-700">
              <Tags className="inline h-4 w-4 mr-2 text-indigo-600" />
              Subtopic
            </label>
            <div className="relative">
              <select
                value={selectedSubtopic}
                onChange={(e) => setSelectedSubtopic(e.target.value)}
                className="w-full p-3 pr-10 text-sm text-gray-700 bg-white border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 transition-all duration-200 appearance-none"
                disabled={loading.subtopics || !selectedSubject}
              >
                <option value="">
                  {loading.subtopics ? 'Loading...' : 'All Subtopics'}
                </option>
                {subtopics.map(subtopic => (
                  <option key={subtopic.id || subtopic} value={subtopic.id || subtopic}>
                    {subtopic.name || subtopic}
                  </option>
                ))}
              </select>
              <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
            </div>
          </div>

          <div className="space-y-3">
            <label className="block text-sm font-semibold text-gray-700">
              <Building className="inline h-4 w-4 mr-2 text-indigo-600" />
              Institution
            </label>
            <div className="relative">
              <select
                value={selectedSchool}
                onChange={(e) => setSelectedSchool(e.target.value)}
                className="w-full p-3 pr-10 text-sm text-gray-700 bg-white border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 transition-all duration-200 appearance-none"
                disabled={loading.schools}
              >
                <option value="">
                  {loading.schools ? 'Loading...' : 'All Institutions'}
                </option>
                {schools.map(school => (
                  <option key={school.id || school} value={school.id || school}>
                    {school.name || school}
                  </option>
                ))}
              </select>
              <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
            </div>
          </div>

          <div className="space-y-3">
            <label className="block text-sm font-semibold text-gray-700">
              <BarChart className="inline h-4 w-4 mr-2 text-indigo-600" />
              Difficulty
            </label>
            <div className="relative">
              <select
                value={selectedDifficulty}
                onChange={(e) => setSelectedDifficulty(e.target.value)}
                className="w-full p-3 pr-10 text-sm text-gray-700 bg-white border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 transition-all duration-200 appearance-none"
              >
                <option value="all">All Difficulties</option>
                {difficulties.map(difficulty => (
                  <option key={difficulty} value={difficulty}>{difficulty}</option>
                ))}
              </select>
              <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
            </div>
          </div>

          <div className="space-y-3">
            <label className="block text-sm font-semibold text-gray-700">
              <Sparkles className="inline h-4 w-4 mr-2 text-indigo-600" />
              Viva Type
            </label>
            <div className="relative">
              <select
                value={selectedVivaType}
                onChange={(e) => setSelectedVivaType(e.target.value)}
                className="w-full p-3 pr-10 text-sm text-gray-700 bg-white border border-gray-300 rounded-lg focus:border-indigo-500 focus:ring-2 focus:ring-indigo-200 transition-all duration-200 appearance-none"
                disabled={loading.vivaTypes}
              >
                <option value="all">
                  {loading.vivaTypes ? 'Loading...' : 'All Viva Types'}
                </option>
                {vivaTypes.map(type => (
                  <option key={type} value={type}>{type}</option>
                ))}
              </select>
              <ChevronDown className="absolute right-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
            </div>
          </div>
        </div>

        <div className="flex flex-col sm:flex-row gap-3 mb-6">
          <button
            onClick={handleFetchQuestions}
            disabled={loading.questions}
            className="flex items-center justify-center px-6 py-3 bg-indigo-600 text-white font-medium rounded-lg hover:bg-indigo-700 focus:ring-4 focus:ring-indigo-200 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {loading.questions ? (
              <>
                <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                Loading Questions...
              </>
            ) : (
              <>
                <Search className="h-4 w-4 mr-2" />
                Fetch Questions
              </>
            )}
          </button>
          
          <button
            onClick={clearFilters}
            disabled={loading.questions}
            className="flex items-center justify-center px-6 py-3 bg-gray-500 text-white font-medium rounded-lg hover:bg-gray-600 focus:ring-4 focus:ring-gray-200 transition-all duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            <X className="h-4 w-4 mr-2" />
            Clear Filters
          </button>
        </div>
        
        <div className="space-y-4">
          {loading.questions ? (
            <div className="text-center py-12">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600 mx-auto mb-4"></div>
              <p className="text-gray-500 text-lg">Loading questions...</p>
            </div>
          ) : filteredQuestions.length > 0 ? (
            <>
              <div className="flex items-center justify-between mb-4 p-3 bg-gray-50 rounded-lg">
                <p className="text-sm text-gray-600">
                  Showing {pagination.from} - {pagination.to} of {pagination.total} questions
                </p>
                <p className="text-sm text-gray-500">
                  Page {pagination.current_page} of {pagination.last_page}
                </p>
              </div>
              
              {filteredQuestions.map(question => (
                <div key={question.id} className="border border-gray-200 rounded-lg p-6 hover:shadow-md hover:border-indigo-200 transition-all duration-300 bg-white">
                  <div className="flex flex-col md:flex-row md:items-start md:justify-between mb-4">
                    <div className="flex flex-wrap items-center gap-2 mb-2 md:mb-0">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        question.difficulty === 'Easy' ? 'bg-green-100 text-green-700' :
                        question.difficulty === 'Medium' ? 'bg-yellow-100 text-yellow-700' :
                        'bg-red-100 text-red-700'
                      }`}>
                        {question.difficulty}
                      </span>
                      <span className="px-2 py-1 bg-blue-100 text-blue-700 rounded-full text-xs font-medium">
                        {question.subject}
                      </span>
                      {question.subtopic && (
                        <span className="px-2 py-1 bg-purple-100 text-purple-700 rounded-full text-xs font-medium">
                          {question.subtopic}
                        </span>
                      )}
                      {question.institution !== 'N/A' && (
                        <span className="px-2 py-1 bg-orange-100 text-orange-700 rounded-full text-xs font-medium">
                          {question.institution}
                        </span>
                      )}
                      {question.viva_type && (
                        <span className="px-2 py-1 bg-pink-100 text-pink-700 rounded-full text-xs font-medium">
                          {question.viva_type}
                        </span>
                      )}
                    </div>
                    <div className="text-xs text-gray-500">
                      By {question.uploadedBy} • {question.uploadedAt}
                    </div>
                  </div>
                  <div className="mb-4">
                    <h3 className="font-semibold text-gray-800 mb-2 flex items-center text-sm">
                      <HelpCircle className="h-4 w-4 mr-2 text-blue-600" />
                      Question:
                    </h3>
                    <p className="text-gray-700 leading-relaxed text-sm">{question.question}</p>
                  </div>
                  <div>
  <h3 className="font-semibold text-gray-800 mb-2 flex items-center text-sm">
    <CheckCircle className="h-4 w-4 mr-2 text-green-600" />
    Answer:
  </h3>
  <div
    className="text-gray-700 leading-relaxed text-sm"
    dangerouslySetInnerHTML={{ __html: question.answer }}
  />
  {question.explanation && (
    <div className="mt-2">
      <h4 className="font-semibold text-gray-800 text-sm">Explanation:</h4>
      <p className="text-gray-600 text-sm">{question.explanation}</p>
    </div>
  )}
</div>

                </div>
              ))}
              
              {pagination.last_page > 1 && (
                <div className="flex items-center justify-between mt-6 p-4 bg-gray-50 rounded-lg">
                  <div className="flex items-center space-x-2">
                    <button
                      onClick={() => handlePageChange(pagination.current_page - 1)}
                      disabled={pagination.current_page === 1 || loading.questions}
                      className="px-3 py-2 text-sm font-medium text-gray-500 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 hover:text-gray-700 disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-200"
                    >
                      <ChevronLeft className="h-4 w-4" />
                    </button>
                    
                    <div className="flex items-center space-x-1">
                      {getPageNumbers().map(page => (
                        <button
                          key={page}
                          onClick={() => handlePageChange(page)}
                          disabled={loading.questions}
                          className={`px-3 py-2 text-sm font-medium rounded-lg transition-all duration-200 ${
                            pagination.current_page === page
                              ? 'bg-indigo-600 text-white'
                              : 'text-gray-500 bg-white border border-gray-300 hover:bg-gray-50 hover:text-gray-700'
                          } disabled:opacity-50 disabled:cursor-not-allowed`}
                        >
                          {page}
                        </button>
                      ))}
                    </div>
                    
                    <button
                      onClick={() => handlePageChange(pagination.current_page + 1)}
                      disabled={pagination.current_page === pagination.last_page || loading.questions}
                      className="px-3 py-2 text-sm font-medium text-gray-500 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 hover:text-gray-700 disabled:opacity-50 disabled:cursor-not-allowed transition-all duration-200"
                    >
                      <ChevronRight className="h-4 w-4" />
                    </button>
                  </div>
                  
                  <div className="hidden sm:block">
                    <p className="text-sm text-gray-600">
                      Go to page:
                    </p>
                    <input
                      type="number"
                      min="1"
                      max={pagination.last_page}
                      value={pagination.current_page}
                      onChange={(e) => {
                        const page = parseInt(e.target.value);
                        if (page >= 1 && page <= pagination.last_page) {
                          handlePageChange(page);
                        }
                      }}
                      className="ml-2 w-16 px-2 py-1 text-sm border border-gray-300 rounded focus:border-indigo-500 focus:ring-1 focus:ring-indigo-500"
                      disabled={loading.questions}
                    />
                  </div>
                </div>
              )}
            </>
          ) : questions.length === 0 && !loading.questions ? (
            <div className="text-center py-12">
              <FileText className="h-16 w-16 text-gray-300 mx-auto mb-4" />
              <p className="text-gray-500 text-lg mb-2">No questions loaded yet</p>
              <p className="text-gray-400">Click "Fetch Questions" to load questions from the database</p>
            </div>
          ) : (
            <div className="text-center py-12">
              <Search className="h-16 w-16 text-gray-300 mx-auto mb-4" />
              <p className="text-gray-500 text-lg mb-2">No questions found</p>
              <p className="text-gray-400">Try adjusting your search criteria or filters</p>
            </div>
          )}
        </div>
      </div>

      {toast.show && (
        <div className={`fixed bottom-4 right-4 p-4 rounded-lg shadow-lg text-white ${
          toast.type === 'error' ? 'bg-red-500' :
          toast.type === 'success' ? 'bg-green-500' :
          'bg-blue-500'
        }`}>
          {toast.message}
        </div>
      )}
    </div>
  );
};

export default ViewQuestions;