import React, { useState, useEffect, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import { ArrowLeft, Volume2, Check, X, Loader2 } from 'lucide-react';
import { toast } from 'react-toastify';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL; 

const FlashcardStudy = ({ cards = [], deckName = "Flashcards" }) => {
  const navigate = useNavigate();

  const [currentIndex, setCurrentIndex] = useState(0);
  const [isFlipped, setIsFlipped] = useState(false);
  const [sessionStats, setSessionStats] = useState({
    studied: 0,
    correct: 0,
    incorrect: 0,
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [reviewStartTime, setReviewStartTime] = useState(Date.now());

  const currentCard = cards[currentIndex];

  const handleFlip = () => {
    setIsFlipped(true);
  };

  const submitReview = async (cardId, rating, timeTaken) => {
    const token = localStorage.getItem('auth_token');
    if (!token) return;

    try {
      const response = await fetch(`${API_BASE_URL}/flashcards/${cardId}/review`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
          'Accept' : 'application/json'
        },
        body: JSON.stringify({
          rating: rating,
          time_taken_seconds: timeTaken,
        }),
      });

      if (response.ok) {
        const data = await response.json();
        console.log('Review submitted:', data);
        return data;
      }
    } catch (error) {
      console.error('Error submitting review:', error);
    }
  };

  const handleRating = useCallback(async (rating) => {
    if (isSubmitting) return;
    
    setIsSubmitting(true);

    // Calculate time taken
    const timeTaken = Math.floor((Date.now() - reviewStartTime) / 1000);

    // Update session stats
    setSessionStats(prev => ({
      ...prev,
      studied: prev.studied + 1,
      correct: rating === 'easy' || rating === 'good' ? prev.correct + 1 : prev.correct,
      incorrect: rating === 'again' || rating === 'hard' ? prev.incorrect + 1 : prev.incorrect,
    }));

    // Submit review to backend
    await submitReview(currentCard.id, rating, timeTaken);

    // Move to next card
    if (currentIndex < cards.length - 1) {
      setCurrentIndex(prev => prev + 1);
      setIsFlipped(false);
      setReviewStartTime(Date.now());
      setIsSubmitting(false);
    } else {
      // End of session - show summary
      setIsSubmitting(false);
      showSessionSummary();
    }
  }, [currentIndex, cards.length, currentCard, reviewStartTime, isSubmitting]);

  const showSessionSummary = () => {
    const accuracy = sessionStats.studied > 0 
      ? Math.round((sessionStats.correct / sessionStats.studied) * 100) 
      : 0;

    toast.success(
      `🎉 Session Complete!\n✓ Studied: ${sessionStats.studied} cards\n📊 Accuracy: ${accuracy}%`,
      { autoClose: 5000 }
    );

    setTimeout(() => {
      navigate('/dashboard/flashcards');
    }, 2000);
  };

  // Keyboard shortcuts
  useEffect(() => {
    const handleKey = (e) => {
      if (isSubmitting) return;

      if (e.key === ' ') {
        e.preventDefault();
        if (!isFlipped) handleFlip();
      } else if (isFlipped) {
        if (e.key === '1') handleRating('again');
        else if (e.key === '2') handleRating('hard');
        else if (e.key === '3') handleRating('good');
        else if (e.key === '4') handleRating('easy');
      }
    };
    window.addEventListener('keydown', handleKey);
    return () => window.removeEventListener('keydown', handleKey);
  }, [isFlipped, handleRating, isSubmitting]);

  // Text-to-speech
  const speak = (text) => {
    if ('speechSynthesis' in window) {
      const utterance = new SpeechSynthesisUtterance(text);
      utterance.rate = 0.9;
      window.speechSynthesis.speak(utterance);
    }
  };

  if (cards.length === 0) {
    return (
      <div className="h-full flex items-center justify-center">
        <p className="text-xl text-gray-500">No cards in this deck yet.</p>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-indigo-50 via-purple-50 to-pink-50 flex flex-col">
      {/* Top Progress Bar - Responsive */}
      <div className="px-4 sm:px-6 py-3 sm:py-4 border-b border-gray-200 bg-white/80 backdrop-blur-sm shadow-sm">
        <div className="max-w-7xl mx-auto">
          {/* Mobile Layout */}
          <div className="flex flex-col gap-3 sm:hidden">
            <div className="flex items-center justify-between">
              <button 
                onClick={() => navigate('/dashboard/flashcards')} 
                className="flex items-center gap-1 text-gray-600 hover:text-gray-900 transition-colors"
              >
                <ArrowLeft size={18} />
                <span className="text-sm font-medium">Back</span>
              </button>
              <h2 className="text-base font-semibold text-gray-900 truncate max-w-[150px]">{deckName}</h2>
              <div className="text-xs text-gray-600 bg-gray-100 px-2 py-1 rounded">
                {sessionStats.studied}/{cards.length}
              </div>
            </div>
            
            {/* Progress Bar Mobile */}
            <div className="w-full">
              <div className="w-full bg-gray-200 rounded-full h-2">
                <div
                  className="bg-gradient-to-r from-indigo-500 to-purple-500 h-2 rounded-full transition-all duration-500"
                  style={{ width: `${((currentIndex + 1) / cards.length) * 100}%` }}
                />
              </div>
              <div className="flex justify-between mt-1">
                <p className="text-xs text-gray-500">
                  Card {currentIndex + 1} of {cards.length}
                </p>
                <p className="text-xs text-gray-500">
                  {Math.round(((currentIndex + 1) / cards.length) * 100)}%
                </p>
              </div>
            </div>
          </div>

          {/* Desktop Layout */}
          <div className="hidden sm:flex items-center justify-between gap-4">
            <button 
              onClick={() => navigate('/dashboard/flashcards')} 
              className="flex items-center gap-2 text-gray-600 hover:text-gray-900 transition-colors"
            >
              <ArrowLeft size={20} />
              <span className="font-medium">Back</span>
            </button>
            
            <h2 className="text-lg font-semibold text-gray-900">{deckName}</h2>

            <div className="flex-1 max-w-md mx-4">
              <div className="w-full bg-gray-200 rounded-full h-3">
                <div
                  className="bg-gradient-to-r from-indigo-500 to-purple-500 h-3 rounded-full transition-all duration-500"
                  style={{ width: `${((currentIndex + 1) / cards.length) * 100}%` }}
                />
              </div>
              <p className="text-sm text-gray-600 text-center mt-2">
                {currentIndex + 1} / {cards.length} cards
              </p>
            </div>

            <div className="flex items-center gap-4">
              <div className="text-sm text-gray-600 flex items-center gap-2">
                <Check size={16} className="text-green-600" />
                {sessionStats.correct}
              </div>
              <div className="text-sm text-gray-600 flex items-center gap-2">
                <X size={16} className="text-red-600" />
                {sessionStats.incorrect}
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Main Card Area - Responsive with proper scrolling */}
      <div className="flex-1 flex items-center justify-center p-3 sm:p-6 overflow-hidden">
        <div className="w-full max-w-5xl h-full flex items-center justify-center">
          {/* Card Container - Fixed height with internal scrolling */}
          <div
            className="relative w-full h-[calc(100vh-180px)] sm:h-[75vh] max-h-[800px]"
            onClick={() => !isFlipped && !isSubmitting && handleFlip()}
          >
            {!isFlipped ? (
              /* FRONT OF CARD */
              <div className="absolute inset-0 transition-all duration-500 transform hover:scale-[1.01]">
                <div className="h-full bg-white rounded-2xl sm:rounded-3xl shadow-xl sm:shadow-2xl p-6 sm:p-10 flex flex-col items-center text-center">
                  {/* Scrollable content area */}
                  <div className="flex-1 overflow-y-auto w-full flex flex-col items-center justify-center">
                    {currentCard.image && (
                      <img 
                        src={currentCard.image} 
                        alt="Question" 
                        className="max-h-48 sm:max-h-64 w-auto rounded-lg mb-6 sm:mb-8 object-contain flex-shrink-0" 
                      />
                    )}
                    <p className="text-lg sm:text-2xl md:text-3xl font-medium text-gray-900 leading-relaxed px-2 mb-6">
                      {currentCard.front}
                    </p>
                    {currentCard.options && Array.isArray(currentCard.options) && currentCard.options.length > 0 && (
                      <div className="mt-4 sm:mt-6 space-y-2 sm:space-y-3 w-full max-w-2xl px-2">
                        {currentCard.options.map((opt, i) => (
                          <div 
                            key={i} 
                            className="py-3 sm:py-3.5 px-4 sm:px-6 bg-gradient-to-r from-gray-50 to-gray-100 rounded-lg sm:rounded-xl text-sm sm:text-base md:text-lg text-left border border-gray-200 hover:border-indigo-300 transition-colors"
                          >
                            {opt}
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                  
                  {/* Fixed bottom instruction */}
                  <div className="mt-6 sm:mt-8 space-y-2 flex-shrink-0">
                    <p className="text-gray-500 text-sm sm:text-base md:text-lg font-medium">
                      Tap or press <kbd className="px-2 py-1 bg-gray-200 rounded text-xs sm:text-sm">Space</kbd> to reveal
                    </p>
                    <p className="text-gray-400 text-xs sm:text-sm">
                      Card {currentIndex + 1} of {cards.length}
                    </p>
                  </div>
                </div>
              </div>
            ) : (
              /* BACK OF CARD */
              <div className="absolute inset-0 transition-all duration-500">
                <div className="h-full bg-white rounded-2xl sm:rounded-3xl shadow-xl sm:shadow-2xl p-6 sm:p-10 flex flex-col">
                  {/* Answer Header - Fixed */}
                  <div className="flex justify-between items-start mb-4 sm:mb-6 flex-shrink-0">
                    <div className="flex-1 pr-4 min-w-0">
                      <div className="inline-block px-3 py-1 bg-green-100 text-green-700 text-xs font-semibold rounded-full mb-2">
                        CORRECT ANSWER
                      </div>
                      <p className="text-xl sm:text-2xl md:text-3xl font-bold text-green-600 break-words">
                        {currentCard.back}
                      </p>
                    </div>
                    {/* <button
                      onClick={(e) => {
                        e.stopPropagation();
                        speak(`${currentCard.back}. ${currentCard.explanation}`);
                      }}
                      className="p-2 sm:p-3 rounded-lg hover:bg-gray-100 transition-colors flex-shrink-0"
                      title="Read aloud"
                    >
                      <Volume2 size={20} className="text-gray-600 sm:w-6 sm:h-6" />
                    </button> */}
                  </div>

                  {/* Scrollable explanation area */}
                  <div className="flex-1 overflow-y-auto mb-4 sm:mb-6 min-h-0">
                    {currentCard.image && (
                      <img 
                        src={currentCard.image} 
                        alt="Explanation" 
                        className="max-h-40 sm:max-h-48 w-auto rounded-lg mb-4 sm:mb-6 object-contain" 
                      />
                    )}
                    <div className="prose prose-sm sm:prose max-w-none">
                      <p className="text-sm sm:text-base md:text-lg text-gray-700 leading-relaxed whitespace-pre-wrap">
                        {currentCard.explanation}
                      </p>
                    </div>
                  </div>

                  {/* Rating Buttons - Fixed at bottom */}
                  <div className="flex-shrink-0">
                    {/* Loading Overlay */}
                    {isSubmitting && (
                      <div className="mb-3 flex items-center justify-center gap-2 text-indigo-600">
                        <Loader2 className="w-5 h-5 animate-spin" />
                        <span className="text-sm font-medium">Saving your progress...</span>
                      </div>
                    )}
                    
                    <div className="grid grid-cols-2 sm:grid-cols-4 gap-2 sm:gap-4">
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          handleRating('again');
                        }}
                        disabled={isSubmitting}
                        className={`py-3 sm:py-5 bg-gradient-to-br from-red-100 to-red-50 text-red-700 rounded-xl sm:rounded-2xl font-semibold hover:from-red-200 hover:to-red-100 transition-all transform hover:scale-105 active:scale-95 shadow-sm ${isSubmitting ? 'opacity-50 cursor-not-allowed' : ''}`}
                      >
                        <span className="block text-sm sm:text-base">Again</span>
                        <span className="hidden sm:block text-xs mt-1 opacity-75">Press 1</span>
                      </button>
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          handleRating('hard');
                        }}
                        disabled={isSubmitting}
                        className={`py-3 sm:py-5 bg-gradient-to-br from-orange-100 to-orange-50 text-orange-700 rounded-xl sm:rounded-2xl font-semibold hover:from-orange-200 hover:to-orange-100 transition-all transform hover:scale-105 active:scale-95 shadow-sm ${isSubmitting ? 'opacity-50 cursor-not-allowed' : ''}`}
                      >
                        <span className="block text-sm sm:text-base">Hard</span>
                        <span className="hidden sm:block text-xs mt-1 opacity-75">Press 2</span>
                      </button>
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          handleRating('good');
                        }}
                        disabled={isSubmitting}
                        className={`py-3 sm:py-5 bg-gradient-to-br from-green-100 to-green-50 text-green-700 rounded-xl sm:rounded-2xl font-semibold hover:from-green-200 hover:to-green-100 transition-all transform hover:scale-105 active:scale-95 shadow-sm ${isSubmitting ? 'opacity-50 cursor-not-allowed' : ''}`}
                      >
                        <span className="block text-sm sm:text-base">Good</span>
                        <span className="hidden sm:block text-xs mt-1 opacity-75">Press 3</span>
                      </button>
                      <button
                        onClick={(e) => {
                          e.stopPropagation();
                          handleRating('easy');
                        }}
                        disabled={isSubmitting}
                        className={`py-3 sm:py-5 bg-gradient-to-br from-indigo-100 to-indigo-50 text-indigo-700 rounded-xl sm:rounded-2xl font-semibold hover:from-indigo-200 hover:to-indigo-100 transition-all transform hover:scale-105 active:scale-95 shadow-sm ${isSubmitting ? 'opacity-50 cursor-not-allowed' : ''}`}
                      >
                        <span className="block text-sm sm:text-base">Easy</span>
                        <span className="hidden sm:block text-xs mt-1 opacity-75">Press 4</span>
                      </button>
                    </div>

                    {/* Mobile Stats Display */}
                    <div className="sm:hidden flex justify-center gap-6 mt-4 pt-4 border-t border-gray-200 text-sm">
                      <div className="flex items-center gap-1 text-green-600">
                        <Check size={16} />
                        <span>{sessionStats.correct} correct</span>
                      </div>
                      <div className="flex items-center gap-1 text-red-600">
                        <X size={16} />
                        <span>{sessionStats.incorrect} wrong</span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default FlashcardStudy;