import React, { useState, useEffect } from 'react';
import { Award, CheckCircle, XCircle, Clock, RefreshCw, Play } from 'lucide-react';
import axios from 'axios';

const Quiz = () => {
  // State management
  const [subjects, setSubjects] = useState([]);
  const [subtopics, setSubtopics] = useState([]);
  const [questions, setQuestions] = useState([]);
  const [currentQuiz, setCurrentQuiz] = useState(null);
  const [currentQuestionIndex, setCurrentQuestionIndex] = useState(0);
  const [userAnswers, setUserAnswers] = useState([]);
  const [quizCompleted, setQuizCompleted] = useState(false);
  const [quizResults, setQuizResults] = useState(null);
  const [showAnswer, setShowAnswer] = useState(false);
  const [timeLeft, setTimeLeft] = useState(null);
  const [quizStarted, setQuizStarted] = useState(false);
  const [toast, setToast] = useState({ show: false, message: '', type: 'info' });

  // Loading states
  const [loading, setLoading] = useState({
    subjects: false,
    subtopics: false,
    questions: false,
    generating: false
  });

  // Quiz settings
  const [quizSettings, setQuizSettings] = useState({
    subject: 'all',
    subtopic: 'all',
    difficulty: 'all',
    questionCount: 10,
    timeLimit: 30 // minutes
  });

  // API configuration
  const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000/api';
  const difficulties = ['Easy', 'Medium', 'Hard'];

  // Toast notification helper
  const showToast = (message, type = 'info') => {
    setToast({ show: true, message, type });
    setTimeout(() => setToast({ show: false, message: '', type: 'info' }), 3000);
  };

  // Fetch subjects on component mount
  useEffect(() => {
    const fetchSubjects = async () => {
      setLoading(prev => ({ ...prev, subjects: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects`);
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubjects(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subjects:', err);
        setSubjects([]);
        showToast('Failed to load subjects', 'error');
      } finally {
        setLoading(prev => ({ ...prev, subjects: false }));
      }
    };
    fetchSubjects();
  }, [API_BASE_URL]);

  // Fetch subtopics when subject changes
  useEffect(() => {
    const fetchSubtopics = async () => {
      if (!quizSettings.subject || quizSettings.subject === 'all') {
        setSubtopics([]);
        setQuizSettings(prev => ({ ...prev, subtopic: 'all' }));
        return;
      }

      setLoading(prev => ({ ...prev, subtopics: true }));
      try {
        const response = await fetch(`${API_BASE_URL}/dashboard/subjects/${quizSettings.subject}/subtopics`);
        if (!response.ok) throw new Error(`HTTP error! Status: ${response.status}`);
        const data = await response.json();
        setSubtopics(data.data || data || []);
      } catch (err) {
        console.error('Error fetching subtopics:', err);
        setSubtopics([]);
        showToast('Failed to load subtopics', 'error');
      } finally {
        setLoading(prev => ({ ...prev, subtopics: false }));
      }
    };
    fetchSubtopics();
  }, [API_BASE_URL, quizSettings.subject]);

  // Fetch questions based on filters
  const fetchQuestions = async (filters = {}) => {
    setLoading(prev => ({ ...prev, questions: true }));
    try {
      const params = new URLSearchParams();
      
      if (filters.subject && filters.subject !== 'all') {
        params.append('subject', filters.subject);
      }
      if (filters.subtopic && filters.subtopic !== 'all') {
        params.append('subtopic', filters.subtopic);
      }
      if (filters.difficulty && filters.difficulty !== 'all') {
        params.append('difficulty', filters.difficulty);
      }
      
      const queryString = params.toString();
      const url = `${API_BASE_URL}/dashboard/get-questions${queryString ? `?${queryString}` : ''}`;
      
      const response = await fetch(url, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
      });
      
      if (!response.ok) {
        throw new Error(`HTTP error! Status: ${response.status}`);
      }
      
      const data = await response.json();
      const questionsData = data.data || data || [];
      
      const transformedQuestions = questionsData.map(q => ({
        id: q.id,
        question: q.question,
        options: q.options || [q.option_a, q.option_b, q.option_c, q.option_d].filter(Boolean),
        correct_answer: q.correct_answer,
        explanation: q.explanation || '',
        subject: q.subject,
        subtopic: q.subtopic,
        difficulty: q.difficulty
      }));
      
      setQuestions(transformedQuestions);
      return transformedQuestions;
    } catch (err) {
      console.error('Error fetching questions:', err);
      setQuestions([]);
      showToast('Failed to load questions', 'error');
      return [];
    } finally {
      setLoading(prev => ({ ...prev, questions: false }));
    }
  };

  // Timer effect
  useEffect(() => {
    let interval;
    if (quizStarted && timeLeft > 0 && !quizCompleted) {
      interval = setInterval(() => {
        setTimeLeft(prev => {
          if (prev <= 1) {
            finishQuiz();
            return 0;
          }
          return prev - 1;
        });
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [quizStarted, timeLeft, quizCompleted]);

  // Generate quiz
  const generateQuiz = async () => {
    setLoading(prev => ({ ...prev, generating: true }));
    try {
      const availableQuestions = await fetchQuestions(quizSettings);
      if (availableQuestions.length === 0) {
        showToast('No questions available for the selected criteria', 'error');
        return;
      }

      const shuffled = [...availableQuestions].sort(() => Math.random() - 0.5);
      const selectedQuestions = shuffled.slice(0, Math.min(quizSettings.questionCount, shuffled.length));
      
      setCurrentQuiz(selectedQuestions);
      setCurrentQuestionIndex(0);
      setUserAnswers(new Array(selectedQuestions.length).fill(''));
      setQuizCompleted(false);
      setQuizResults(null);
      setShowAnswer(false);
      setTimeLeft(quizSettings.timeLimit * 60);
      setQuizStarted(true);
      
      showToast(`Quiz started with ${selectedQuestions.length} questions!`, 'success');
    } catch (err) {
      console.error('Error generating quiz:', err);
      showToast('Failed to generate quiz', 'error');
    } finally {
      setLoading(prev => ({ ...prev, generating: false }));
    }
  };

  // Handle answer selection
  const handleAnswerSelect = (selectedOption) => {
    const newAnswers = [...userAnswers];
    newAnswers[currentQuestionIndex] = selectedOption;
    setUserAnswers(newAnswers);
  };

  // Next question
  const nextQuestion = () => {
    if (currentQuestionIndex < currentQuiz.length - 1) {
      setCurrentQuestionIndex(prev => prev + 1);
      setShowAnswer(false);
    } else {
      finishQuiz();
    }
  };

  // Previous question
  const prevQuestion = () => {
    if (currentQuestionIndex > 0) {
      setCurrentQuestionIndex(prev => prev - 1);
      setShowAnswer(false);
    }
  };

  const saveResult = async (resultData) => {
    try {
      const response = await axios.post(
        `${API_BASE_URL}/dashboard/quiz/save-result`,
        resultData,
        {
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('token')}`,
            'Content-Type': 'application/json'
          }
        }
      );
      return response.data;
    } catch (error) {
      console.error('Error saving quiz result:', error);
      throw error;
    }
  };

  const finishQuiz = async () => {
    if (!currentQuiz) return;

    const correctAnswers = currentQuiz.reduce((count, question, index) => {
      const userAnswerText = userAnswers[index];
      const correctAnswerLetter = question.correct_answer;
      const userAnswerLetter = getUserAnswerLetter(userAnswerText, question.options);
      return userAnswerLetter === correctAnswerLetter ? count + 1 : count;
    }, 0);

    const results = {
      userData: JSON.parse(localStorage.getItem('user_data')),
      totalQuestions: currentQuiz.length,
      correctAnswers,
      incorrectAnswers: currentQuiz.length - correctAnswers,
      score: Math.round((correctAnswers / currentQuiz.length) * 100),
      questions: currentQuiz,
      timeSpent: quizSettings.timeLimit * 60 - timeLeft
    };

    try {
      await saveResult(results);
    } catch (error) {
      console.error('Failed to save quiz results.');
    }

    setQuizResults(results);
    setQuizCompleted(true);
    setQuizStarted(false);
  };

  // Helper function to convert text answer to letter
  const getUserAnswerLetter = (userAnswerText, options) => {
    const optionIndex = options.findIndex(option => option === userAnswerText);
    if (optionIndex === -1) return null;
    return String.fromCharCode(65 + optionIndex);
  };

  // Reset quiz
  const resetQuiz = () => {
    setCurrentQuiz(null);
    setCurrentQuestionIndex(0);
    setUserAnswers([]);
    setQuizCompleted(false);
    setQuizResults(null);
    setShowAnswer(false);
    setTimeLeft(null);
    setQuizStarted(false);
  };

  // Format time
  const formatTime = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
  };

  // Helper function to convert letter to text answer
  const getCorrectAnswerText = (correctAnswerLetter, options) => {
    const letterIndex = correctAnswerLetter.charCodeAt(0) - 65;
    return options[letterIndex];
  };

  // Results view
  if (quizCompleted && quizResults) {
    return (
      <div className="space-y-6">
        <h1 className="text-3xl font-bold text-gray-800">Quiz Results</h1>
        <div className="bg-white p-6 rounded-lg shadow-md">
          <div className="text-center mb-6">
            <div className={`w-24 h-24 mx-auto mb-4 rounded-full flex items-center justify-center ${
              quizResults.score >= 70 ? 'bg-green-100' : quizResults.score >= 50 ? 'bg-yellow-100' : 'bg-red-100'
            }`}>
              <Award className={`${
                quizResults.score >= 70 ? 'text-green-600' : quizResults.score >= 50 ? 'text-yellow-600' : 'text-red-600'
              }`} size={48} />
            </div>
            <h2 className="text-2xl font-bold text-gray-800 mb-2">Quiz Completed!</h2>
            <p className="text-gray-600">You scored {quizResults.score}% ({quizResults.correctAnswers}/{quizResults.totalQuestions})</p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
            <div className="bg-blue-50 p-4 rounded-lg text-center">
              <p className="text-sm text-blue-600">Total Questions</p>
              <p className="text-2xl font-bold text-blue-800">{quizResults.totalQuestions}</p>
            </div>
            <div className="bg-green-50 p-4 rounded-lg text-center">
              <p className="text-sm text-green-600">Correct</p>
              <p className="text-2xl font-bold text-green-800">{quizResults.correctAnswers}</p>
            </div>
            <div className="bg-red-50 p-4 rounded-lg text-center">
              <p className="text-sm text-red-600">Incorrect</p>
              <p className="text-2xl font-bold text-red-800">{quizResults.incorrectAnswers}</p>
            </div>
            <div className="bg-purple-50 p-4 rounded-lg text-center">
              <p className="text-sm text-purple-600">Score</p>
              <p className="text-2xl font-bold text-purple-800">{quizResults.score}%</p>
            </div>
          </div>

          <div className="space-y-4 mb-6">
            <h3 className="text-lg font-semibold">Review Your Answers:</h3>
            {quizResults.questions.map((question, index) => {
              const userAnswerText = userAnswers[index];
              const correctAnswerLetter = question.correct_answer;
              const correctAnswerText = getCorrectAnswerText(correctAnswerLetter, question.options);
              const isCorrect = userAnswerText === correctAnswerText;

              return (
                <div key={question.id} className="border border-gray-200 rounded-lg p-4">
                  <div className="flex items-start justify-between mb-2">
                    <span className="font-medium text-sm">Question {index + 1}</span>
                    <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                      isCorrect ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                    }`}>
                      {isCorrect ? 'Correct' : 'Incorrect'}
                    </span>
                  </div>
                  <p className="text-gray-700 mb-3">{question.question}</p>

                  <div className="space-y-2 mb-3">
                    {question.options.map((option, optIndex) => {
                      const optionLetter = String.fromCharCode(65 + optIndex);
                      return (
                        <div key={optIndex} className={`p-2 rounded border ${
                          option === correctAnswerText ? 'bg-green-50 border-green-200' :
                          option === userAnswerText && !isCorrect ? 'bg-red-50 border-red-200' :
                          'bg-gray-50 border-gray-200'
                        }`}>
                          <div className="flex items-center space-x-2">
                            <span className="font-medium">{optionLetter}.</span>
                            <span>{option}</span>
                            {option === correctAnswerText && (
                              <CheckCircle className="text-green-600" size={16} />
                            )}
                            {option === userAnswerText && !isCorrect && (
                              <XCircle className="text-red-600" size={16} />
                            )}
                          </div>
                        </div>
                      );
                    })}
                  </div>

                  {question.explanation && (
                    <div className="bg-blue-50 p-3 rounded">
                      <p className="text-sm text-blue-600 mb-1">Explanation:</p>
                      <p className="text-blue-800">{question.explanation}</p>
                    </div>
                  )}
                </div>
              );
            })}
          </div>

          <div className="flex justify-center">
            <button
              onClick={resetQuiz}
              className="bg-blue-500 text-white px-6 py-2 rounded-md hover:bg-blue-600 transition-colors flex items-center space-x-2"
            >
              <RefreshCw size={20} />
              <span>Take Another Quiz</span>
            </button>
          </div>
        </div>
      </div>
    );
  }

  // Quiz in progress
  if (currentQuiz && !quizCompleted) {
    const currentQuestion = currentQuiz[currentQuestionIndex];
    const progress = ((currentQuestionIndex + 1) / currentQuiz.length) * 100;

    return (
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <h1 className="text-3xl font-bold text-gray-800">Quiz in Progress</h1>
          <div className="flex items-center space-x-4">
            <div className="flex items-center space-x-2 text-red-600">
              <Clock size={20} />
              <span className="font-bold">{formatTime(timeLeft)}</span>
            </div>
            <button
              onClick={resetQuiz}
              className="text-red-500 hover:text-red-700 flex items-center space-x-2"
            >
              <XCircle size={20} />
              <span>Exit Quiz</span>
            </button>
          </div>
        </div>

        <div className="bg-white p-6 rounded-lg shadow-md">
          <div className="mb-6">
            <div className="flex justify-between items-center mb-2">
              <span className="text-sm text-gray-600">Question {currentQuestionIndex + 1} of {currentQuiz.length}</span>
              <span className="text-sm text-gray-600">{Math.round(progress)}% Complete</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div 
                className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                style={{ width: `${progress}%` }}
              ></div>
            </div>
          </div>

          <div className="mb-6">
            <div className="flex flex-wrap gap-2 mb-4">
              <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                currentQuestion.difficulty === 'Easy' ? 'bg-green-100 text-green-800' :
                currentQuestion.difficulty === 'Medium' ? 'bg-yellow-100 text-yellow-800' :
                'bg-red-100 text-red-800'
              }`}>
                {currentQuestion.difficulty}
              </span>
              <span className="px-2 py-1 bg-blue-100 text-blue-800 rounded-full text-xs font-medium">
                {currentQuestion.subject}
              </span>
              {currentQuestion.subtopic && (
                <span className="px-2 py-1 bg-purple-100 text-purple-800 rounded-full text-xs font-medium">
                  {currentQuestion.subtopic}
                </span>
              )}
            </div>

            <h3 className="text-lg font-semibold mb-4">Question:</h3>
            <p className="text-gray-700 mb-6">{currentQuestion.question}</p>

            <div className="space-y-2 mb-6">
              {currentQuestion.options.map((option, index) => (
                <button
                  key={index}
                  onClick={() => handleAnswerSelect(option)}
                  className={`w-full p-3 text-left rounded-lg border transition-all ${
                    userAnswers[currentQuestionIndex] === option
                      ? 'bg-blue-50 border-blue-300 text-blue-800'
                      : 'bg-gray-50 border-gray-200 hover:bg-gray-100'
                  }`}
                >
                  <div className="flex items-center space-x-3">
                    <span className="flex-shrink-0 w-6 h-6 rounded-full border border-gray-300 flex items-center justify-center text-sm font-medium">
                      {String.fromCharCode(65 + index)}
                    </span>
                    <span>{option}</span>
                  </div>
                </button>
              ))}
            </div>

            {showAnswer && (
              <div className="bg-green-50 p-4 rounded-lg mb-4">
                <h4 className="font-semibold text-green-800 mb-2">Correct Answer:</h4>
                <p className="text-green-700">{getCorrectAnswerText(currentQuestion.correct_answer, currentQuestion.options)}</p>
                {currentQuestion.explanation && (
                  <div className="mt-2">
                    <p className="text-sm text-green-600 mb-1">Explanation:</p>
                    <p className="text-green-700">{currentQuestion.explanation}</p>
                  </div>
                )}
              </div>
            )}
          </div>

          <div className="flex justify-between">
            <div className="flex space-x-2">
              <button
                onClick={prevQuestion}
                disabled={currentQuestionIndex === 0}
                className="bg-gray-500 text-white px-4 py-2 rounded-md hover:bg-gray-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Previous
              </button>
              <button
                onClick={() => setShowAnswer(!showAnswer)}
                className="bg-yellow-500 text-white px-4 py-2 rounded-md hover:bg-yellow-600 transition-colors"
              >
                {showAnswer ? 'Hide Answer' : 'Show Answer'}
              </button>
            </div>

            <button
              onClick={nextQuestion}
              className="bg-blue-500 text-white px-6 py-2 rounded-md hover:bg-blue-600 transition-colors"
            >
              {currentQuestionIndex === currentQuiz.length - 1 ? 'Finish Quiz' : 'Next Question'}
            </button>
          </div>
        </div>
      </div>
    );
  }

  // Quiz setup
  return (
    <div className="space-y-6">
      <h1 className="text-3xl font-bold text-gray-800">Create Quiz</h1>
      <div className="bg-white p-6 rounded-lg shadow-md">
        <div className="mb-6">
          <h2 className="text-xl font-semibold mb-4">Quiz Settings</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Subject</label>
              <select
                value={quizSettings.subject}
                onChange={(e) => setQuizSettings({...quizSettings, subject: e.target.value})}
                className="w-full p-2 border border-gray-300 rounded-md"
                disabled={loading.subjects}
              >
                <option value="all">All Subjects</option>
                {subjects.map(subject => (
                  <option key={subject.id || subject} value={subject.id || subject}>
                    {subject.name || subject}
                  </option>
                ))}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Subtopic</label>
              <select
                value={quizSettings.subtopic}
                onChange={(e) => setQuizSettings({...quizSettings, subtopic: e.target.value})}
                className="w-full p-2 border border-gray-300 rounded-md"
                disabled={loading.subtopics || quizSettings.subject === 'all'}
              >
                <option value="all">All Subtopics</option>
                {subtopics.map(subtopic => (
                  <option key={subtopic.id || subtopic} value={subtopic.id || subtopic}>
                    {subtopic.name || subtopic}
                  </option>
                ))}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Difficulty</label>
              <select
                value={quizSettings.difficulty}
                onChange={(e) => setQuizSettings({...quizSettings, difficulty: e.target.value})}
                className="w-full p-2 border border-gray-300 rounded-md"
              >
                <option value="all">All Difficulties</option>
                {difficulties.map(difficulty => (
                  <option key={difficulty} value={difficulty}>{difficulty}</option>
                ))}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Number of Questions</label>
              <select
                value={quizSettings.questionCount}
                onChange={(e) => setQuizSettings({...quizSettings, questionCount: parseInt(e.target.value)})}
                className="w-full p-2 border border-gray-300 rounded-md"
              >
                <option value={5}>5 Questions</option>
                <option value={10}>10 Questions</option>
                <option value={15}>15 Questions</option>
                <option value={20}>20 Questions</option>
                <option value={25}>25 Questions</option>
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Time Limit (minutes)</label>
              <select
                value={quizSettings.timeLimit}
                onChange={(e) => setQuizSettings({...quizSettings, timeLimit: parseInt(e.target.value)})}
                className="w-full p-2 border border-gray-300 rounded-md"
              >
                <option value={1}>1 Minute</option>
                <option value={15}>15 Minutes</option>
                <option value={30}>30 Minutes</option>
                <option value={45}>45 Minutes</option>
                <option value={60}>1 Hour</option>
                <option value={90}>1.5 Hours</option>
              </select>
            </div>
          </div>
        </div>

        <button
          onClick={generateQuiz}
          disabled={loading.generating}
          className="bg-blue-500 text-white px-6 py-3 rounded-md hover:bg-blue-600 transition-colors flex items-center space-x-2 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          {loading.generating ? (
            <>
              <RefreshCw className="animate-spin" size={20} />
              <span>Generating Quiz...</span>
            </>
          ) : (
            <>
              <Play size={20} />
              <span>Start Quiz</span>
            </>
          )}
        </button>

        <div className="mt-6">
          <h3 className="text-lg font-semibold mb-4">Available Questions</h3>
          {loading.subjects ? (
            <div className="text-center py-4">Loading subjects...</div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {subjects.map(subject => (
                <div key={subject.id || subject} className="bg-gray-50 p-4 rounded-lg">
                  <h4 className="font-medium text-gray-800">{subject.name || subject}</h4>
                  <p className="text-sm text-gray-600">
                    {subject.questions_count || '0'} questions available
                  </p>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default Quiz;